/*
 * Java
 *
 * Copyright 2021-2023 MicroEJ Corp. All rights reserved.
 * MicroEJ Corp. PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.is2t.s3.io;

import java.io.IOException;
import java.io.InputStream;

import com.is2t.tools.ArrayTools;

/**
 * InputStream returned by {@link Class#getResourceAsStream(String)}.
 */
public class ResourceInputStream extends InputStream {
	
	private final int handle; // VM Known
	
	private boolean closed;
	
	public ResourceInputStream(int handle) {
		this.handle = handle;
	}
	
	@Override
	public int available() throws IOException {
		if (closed) {
			return 0;
		}
		return availableResource(this.handle);
	}
	
	@Override
	public int read() throws IOException {
		if (closed) {
			throw new IOException();
		}
		return readResource(this.handle);
	}
	
	@Override
	public int read(byte[] b, int off, int len) throws IOException {
		if (closed) {
			throw new IOException();
		}
		ArrayTools.checkBounds(b, off, len);
		return readResourceBuffer(handle, b, off, len);
	}
	
	@Override
	public boolean markSupported() {
		return true;
	}
	
	@Override
	public synchronized void mark(int readlimit) {
		if (closed) {
			return;
		}
		markResource(handle, readlimit);
	}
	
	@Override
	public synchronized void reset() throws IOException {
		if (closed) {
			throw new IOException();
		}
		int ret = resetResource(handle);
		if (ret != 0) {
			throw new IOException();
		}
	}
	
	@Override
	public void close() throws IOException {
		if (closed) {
			return;
		}
		closeResource(this.handle);
		this.closed = true;
	}
    
	native static private int availableResource(int handle);

	native static private int readResource(int handle);
	
	native static private int readResourceBuffer(int handle, byte[] buffer, int off, int len);
	
	native static private void closeResource(int hande);
	
	native static private void markResource(int handle, int readLimit);
	
	native static private int resetResource(int handle);
}

