/*
 * Java
 *
 * Copyright 2009-2019 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.fp;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;

/**
 * Widget is the superclass of all the user interface objects. It represents a part of a device that can be graphically
 * represented.
 * <p>
 * A widget is created by the front panel parser. The parser sets its characteristics (configuration step) and ask the
 * widget to check its own configuration ({@link #finalizeConfiguration()}. When all widgets in the hierarchy have been
 * created and checked, each widget is started ({@link #start()}.
 * <p>
 * The parser is using the widget description ({@link WidgetDescription}) to retrieve the widget itself and the widget
 * characteristics the parser can set. Each widget concrete subclass must define its own {@link WidgetDescription} in
 * order to be recognized by the parser. This description must be written in the class java doc.
 * <p>
 * Example:
 *
 * <pre>
 * <code>
 * /**
 *  * @WidgetDescription(
 *  *	attributes = {
 *  *		@WidgetAttribute(name = "label"),
 *  *		@WidgetAttribute(name = "x"),
 *  *		@WidgetAttribute(name = "y"),
 *  *		@WidgetAttribute(name = "ledOff"),
 *  *		@WidgetAttribute(name = "ledOn"),
 *  *		@WidgetAttribute(name = "overlay", isOptional = true)
 *  *	}
 *  * )
 *  *&#47;
 *  public class LED extends Widget {
 *  	// ...
 *  }
 * </code>
 * </pre>
 *
 * This description allows the user to use this widget declaration in his/her front panel file (<code>*.fp</code>):
 *
 * <pre>
 * &lt;ej.fp.widget.LED label="3" x="140" y="753" ledOff="Led-0.png" ledOn="Led-BLUE.png" overlay="false"/&gt;
 * </pre>
 *
 * As soon as a widget holds a description, the parser is able to load the corresponded class. The class is retrieved
 * thanks the widget declaration in the front panel file: <code>package.WidgetClassName</code>. The widget constructor
 * must be empty (no argument).
 * <p>
 * A {@link WidgetAttribute} allows the parser to call the corresponded method <code>setXxx()</code>, where
 * <code>Xxx</code> id the {@link WidgetAttribute} name field, to configure a widget characteristic.
 * <p>
 * This method must exist in the widget hierarchy. If not, an error during parsing will be thrown to stop the execution.
 * <p>
 * Some <code>setXxx()</code> methods can exist in the widget hierarchy whereas they are not described in
 * {@link WidgetDescription} of this widget. The parser will never call these methods. These methods and the associated
 * attributes are optional; it is the responsability to the widget to add (and use) them in its own description.
 */
@SuppressWarnings("nls")
public abstract class Widget {

	/**
	 * This interface allows a subclass of {@link Widget} to describe itself. The front panel parser is using this
	 * description to recognize the widget and its characteristics.
	 */
	@Retention(RetentionPolicy.RUNTIME)
	public static @interface WidgetDescription {

		/**
		 * Lists the widget attributes the front panel parser can set / configure.
		 *
		 * @return widget attributes (or <i>characteristics</i>).
		 */
		WidgetAttribute[] attributes();
	}

	/**
	 * This interface describes only one attribute of the widget description.
	 */
	@Retention(RetentionPolicy.RUNTIME)
	public static @interface WidgetAttribute {

		/**
		 * Sets the name of the attribute. This name is used in front panel file (<code>*.fp</code>) and in widget
		 * hierarchy as method name: <code>setXxx()</code> where <code>Xxx</code> is this attribute name value.
		 *
		 * @return the attribute name.
		 */
		String name();

		/**
		 * Defines the attribute as optional. By default an attribute is mandatory and the parser throws an error when a
		 * mandatory attribute is missing. When optional, the widget instance has to choose a default value.
		 *
		 * @return true when the attribute is optional.
		 */
		boolean isOptional() default false;
	}

	/**
	 * Default widget label when no label is defined by front panel file.
	 */
	public static final String DEFAULT_LABEL = "";

	// widget characteristics (null-init, zero-init or false-init)
	private String label = DEFAULT_LABEL; // never null
	private int x;
	private int y;
	private int width;
	private int height;
	private Image skin;
	private Image filter;
	private boolean overlay;

	// widget hierarchy; when null, the widget cannot be displayed
	private Composite composite; // null-init

	// current visual representation of widget; when null, nothing is displayed
	private Image currentSkin; // null-init

	/**
	 * Creates a widget.
	 * <p>
	 * By default:
	 * <ul>
	 * <li>its bounds are <code>0</code>,</li>
	 * <li>it is not visible,</li>
	 * <li>its label is "" (empty string).</li>
	 * </ul>
	 */
	public Widget() {
	}

	/**
	 * Sets the label (<i>tag</i>) of this widget. When null or not set, the default string "" (empty string) is used.
	 * This label is useful to retrieve a specific widget in the device hierarchy.
	 * <p>
	 * This method should only be called by front panel parser.
	 *
	 * @param label
	 *            the widget label.
	 * @see Device#getWidget(Class, String)
	 */
	public void setLabel(String label) {
		this.label = label == null ? DEFAULT_LABEL : label;
	}

	/**
	 * Sets the x relative coordinate of this widget. The coordinate is relative to the widget's composite position.
	 * <p>
	 * Default position is <code>0,0</code>.
	 * <p>
	 * This method should only be called by front panel parser.
	 *
	 * @param x
	 *            the x coordinate to set.
	 */
	public void setX(int x) {
		this.x = x;
	}

	/**
	 * Sets the y relative coordinate of this widget. The coordinate is relative to the widget's composite position.
	 * <p>
	 * Default position is <code>0,0</code>.
	 * <p>
	 * This method should only be called by front panel parser.
	 *
	 * @param y
	 *            the y coordinate to set.
	 */
	public void setY(int y) {
		this.y = y;
	}

	/**
	 * Sets the width of widget.
	 *
	 * @param width
	 *            the width coordinate to set.
	 * @throws IllegalArgumentException
	 *             when the width is negative.
	 */
	public void setWidth(int width) {
		if (width < 0) {
			throw new IllegalArgumentException("Widget width cannot be negative.");
		}
		this.width = width;
	}

	/**
	 * Sets the height of this widget.
	 *
	 * @param height
	 *            the height coordinate to set.
	 * @throws IllegalArgumentException
	 *             when the height is negative.
	 */
	public void setHeight(int height) {
		if (height < 0) {
			throw new IllegalArgumentException("Widget height cannot be negative.");
		}
		this.height = height;
	}

	/**
	 * Sets the default skin of this widget.
	 * <p>
	 * The skin defines too the widget bounds (width and height). It may be null, in this case the widget will be not
	 * rendered.
	 * <p>
	 * When not null, the bounds optionaly set by {@link #setWidth(int)} and {@link #setHeight(int)} will be replaced by
	 * skin bounds during the widget configuration ({@link #finalizeConfiguration()}).
	 * <p>
	 * Default skin is <code>null</code>.
	 * <p>
	 * This method should only be called by front panel parser.
	 *
	 * @param skin
	 *            the skin to set.
	 */
	public void setSkin(Image skin) {
		this.skin = skin;
	}

	/**
	 * Sets an image which defines a filtering area. The method {@link #isOver(int, int)} returns false if the given
	 * coordinates are <i>inside</i> the skin area but <i>outside</i> the filtering area.
	 * <p>
	 * The filter image and the widget size must have the same size. If not, an error will be thrown when the parser
	 * will check the widget configuration ({@link #finalizeConfiguration()}).
	 * <p>
	 * The filtering area is defined by the pixels whose color is not fully transparent (alpha > 0).
	 * <p>
	 * By default there is no filtering area and all pixels in skin are considered <i>inside</i> the filtering area.
	 * <p>
	 * This method should only be called by front panel parser.
	 *
	 * @param filter
	 *            the image which defines the filtering area.
	 */
	public void setFilter(Image filter) {
		this.filter = filter;
	}

	/**
	 * Sets the repaint policy. If <code>true</code>, the current skin ({@link #getCurrentSkin()} is drawn over the
	 * existent image; otherwise the background and the other widgets behind are drawn before rendering this widget.
	 * <p>
	 * By default, overlay is <code>false</code>.
	 * <p>
	 * This method should only be called by front panel parser.
	 *
	 * @param overlay
	 *            the repaint policy.
	 * @see Widget#repaint()
	 * @see Widget#repaint(int, int, int, int)
	 */
	public final void setOverlay(boolean overlay) {
		this.overlay = overlay;
	}

	/**
	 * Checks whether the widget configuration is valid or not and finalizes the widget configuration according all
	 * attributes previously set.
	 * <p>
	 * The widget attributes are set by the front panel parser but the order is not defined. This method allows to
	 * perform some checks between several attributes (this method is called by the front panel parser after setting all
	 * attributes defined in front panel file).
	 * <p>
	 * The configuration validation depends on widget itself. Each subclass must verify its configuration after checking
	 * its parent configuration.
	 * <p>
	 * When framework calling this method, all device widgets are not created and finalized yet, and by extension the
	 * device itself is not fully created too. So the implementation must not have to call some methods on other widgets
	 * and the methods {@link Device#getDevice()} or {@link FrontPanel#getDevice()}. As soon as all widgets are created
	 * and finalized, the simulation starts calling the method {@link #start()} on each widget. As this moment only, the
	 * widget is able to <i>use</i> the other widgets and the device itself.
	 * <p>
	 * On configuration error, a {@link RuntimeException} (or a subclass of {@link RuntimeException}) must be thrown.
	 * <p>
	 * This method should only be called by front panel parser.
	 *
	 * @throws IllegalStateException
	 *             when a filter has been set whereas there is no skin or when skin and filter bounds are not equal.
	 */
	public synchronized void finalizeConfiguration() {

		// fix widget bounds (zero-init by default)
		Image skin = getSkin();
		if (skin != null) {
			this.width = skin.getWidth();
			this.height = skin.getHeight();
		}

		// check filter: a skin is mandatory and must have the same sizes
		Image filter = this.filter;
		if (filter != null && (getHeight() != filter.getHeight() || getWidth() != filter.getWidth())) {
			throw new IllegalStateException("Filter image must have the same size than widget itself.");
		}

		// skin to show is the default skin on startup
		// (do not call setCurrentSkin() which requires the simulation is started)
		this.currentSkin = this.skin;
	}

	/**
	 * Called by front panel viewer to make widgets appear as if "switched on" or not. Useful to render widgets during
	 * the front panel device creation: it allows to show some "visual" widgets (display, LEDs etc.) as at runtime.
	 * <p>
	 * This method does nothing by default.
	 * <p>
	 * This method should only be called by front panel engine.
	 *
	 * @param appearSwitchedOn
	 *            true to switch on widget, false to switch off.
	 */
	public void showYourself(boolean appearSwitchedOn) {
		// nothing to do by default
	}

	/**
	 * Starts the widget. For instance a widget can start a thread, call methods on other widgets (all widgets are
	 * created and finalized, see {@link #finalizeConfiguration()}).
	 * <p>
	 * Main implementation does nothing.
	 * <p>
	 * This method should only be called by front panel engine.
	 */
	public void start() {
		// nothing to do by default
	}

	/**
	 * Disposes the widget's associated resources (such as threads, images...).
	 * <p>
	 * Default behavior disposes the common images and subclasses should override this method to define their own
	 * behavior.
	 * <p>
	 * This method should only be called by front panel engine.
	 */
	public void dispose() {
		FrontPanel.getFrontPanel().disposeIfNotNull(this.skin, this.currentSkin, this.filter);
		this.skin = null;
		this.currentSkin = null;
		this.filter = null;
	}

	/**
	 * Gets the label (default label {@link #DEFAULT_LABEL} or the label set by the front panel parser thanks
	 * {@link #setLabel(String)} method).
	 *
	 * @return the widget label.
	 */
	public String getLabel() {
		return this.label;
	}

	/**
	 * Gets the default skin of this widget set by the front panel parser thanks {@link #setSkin(Image)} method).
	 *
	 * @return the default skin.
	 */
	public Image getSkin() {
		return this.skin;
	}

	/**
	 * Gets the relative x coordinate of this widget set by the front panel parser thanks {@link #setX(int)} method).
	 *
	 * @return the relative x coordinate.
	 */
	public int getX() {
		return this.x;
	}

	/**
	 * Gets the relative y coordinate of this widget set by the front panel parser thanks {@link #setY(int)} method).
	 *
	 * @return the relative y coordinate.
	 */
	public int getY() {
		return this.y;
	}

	/**
	 * Gets the width of this widget set by the front panel parser thanks {@link #setSkin(Image)} method).
	 *
	 * @return the width.
	 */
	public int getWidth() {
		return this.width;
	}

	/**
	 * Gets the height of this widget set by the front panel parser thanks {@link #setSkin(Image)} method).
	 *
	 * @return the height.
	 */
	public int getHeight() {
		return this.height;
	}

	/**
	 * Gets the image which represents the filtering area set by the front panel parser thanks {@link #setFilter(Image)}
	 * method).
	 *
	 * @return the filtering area image.
	 */
	public Image getFilter() {
		return this.filter;
	}

	/**
	 * Tells whether the specified point is inside the bounds of the wanted form. In other terms, if this widget must
	 * receive an event when the mouse is over this point.
	 * <p>
	 * The point is relative to the composite of this widget.
	 * <p>
	 * By default, return <code>true</code> if the point is inside the bounds of the widget (rectangle) AND id the point
	 * is inside the filtering area (@see {@link #setFilter(Image)}).
	 *
	 * @param x
	 *            the x coordinates of the point to check.
	 * @param y
	 *            the y coordinates of the point to check.
	 * @return true if the point is over this widget, false otherwise.
	 */
	public boolean isOver(int x, int y) {
		return isOverWidget(x, y) && isOverFilter(x, y);
	}

	/**
	 * Gets the absolute x coordinate of this widget.
	 *
	 * @return the absolute x coordinate of this widget.
	 */
	public int getAbsoluteX() {
		Composite parent = getParent();
		return parent == null ? this.x : parent.getAbsoluteX() + this.x;
	}

	/**
	 * Gets the absolute y coordinate of this widget.
	 *
	 * @return the absolute y coordinate of this widget.
	 */
	public int getAbsoluteY() {
		Composite parent = getParent();
		return parent == null ? this.y : parent.getAbsoluteY() + this.y;
	}

	/**
	 * Gets the parent ({@link Composite}) of this widget.
	 *
	 * @return the parent of this widget.
	 */
	public Composite getParent() {
		return this.composite;
	}

	/**
	 * Gets the current skin to render.
	 * <p>
	 * This skin can change at runtime to visualize the widget state. For instance a button has at least two states:
	 * pressed and released. Released state is often the default state so the widget skin ({@link #setSkin(Image)})
	 * should represent the button released. On button pressed, this method should return another image which represents
	 * the button pressed.
	 * <p>
	 * Note that the returned image can be higher or smaller than the widget size. The framework will perform an image
	 * deformation.
	 *
	 * @return the current skin to render.
	 * @see #setCurrentSkin(Image)
	 */
	public Image getCurrentSkin() {
		return this.currentSkin;
	}

	/**
	 * Sets the current skin to render according the widget state.
	 * <p>
	 * This skin will be rendered on the device. If the skin is null, the receiver will be invisible. The method
	 * {@link #repaint()} is called just after the update to force to render the new currebt skin.
	 *
	 * @param skin
	 *            the new skin to render.
	 * @see #getCurrentSkin()
	 */
	public void setCurrentSkin(Image skin) {
		this.currentSkin = skin;
		repaint();
	}

	/**
	 * Asks a repaint of the receiver on the device. The effective repaint is platform dependent. Framework cannot
	 * ensure the result nor the synchronization.
	 */
	public void repaint() {
		if (this.overlay) {
			// repaint only the widget
			FrontPanel.getFrontPanel().repaintWidget(this);
		} else {
			// have to repaint behind the widget
			FrontPanel.getFrontPanel().repaintDevice(getAbsoluteX(), getAbsoluteY(), this.width, this.height);
		}
	}

	/**
	 * Asks a repaint of a part of the receiver on the device. The effective repaint is platform dependent. Framework
	 * cannot ensure the result nor the synchronization.
	 *
	 * @param x
	 *            x coordinate of area to repaint
	 * @param y
	 *            y coordinate of area to repaint
	 * @param width
	 *            width of area to repaint
	 * @param height
	 *            height of area to repaint
	 */
	public void repaint(int x, int y, int width, int height) {
		if (this.overlay) {
			// repaint only the widget
			FrontPanel.getFrontPanel().repaintWidget(this, x, y, width, height);
		} else {
			// have to repaint behind the widget
			FrontPanel.getFrontPanel().repaintDevice(getAbsoluteX() + x, getAbsoluteY() + y, width, height);
		}
	}

	/**
	 * Updates the widget parent. The parent may be null, which means the widget will not be rendered anymore.
	 *
	 * @param composite
	 *            the new parent or null.
	 */
	/* package */ void setParent(Composite composite) {
		this.composite = composite;
	}

	private boolean isOverWidget(int x, int y) {
		return this.x < x && this.y < y && this.x + this.width > x && this.y + this.height > y;
	}

	private boolean isOverFilter(int x, int y) {
		try {
			Image filter = this.filter;
			return filter == null || ((filter.readPixel(x - getAbsoluteX(), y - getAbsoluteY()) & 0xff000000) != 0);
		} catch (IndexOutOfBoundsException e) {
			return false;
		}
	}
}
