/*
 * Copyright 2014-2017 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.motion;

import ej.bon.Util;

/**
 * Common abstract implementation of a motion.
 */
public abstract class AbstractMotion implements Motion {

	/**
	 * Start value.
	 */
	protected final int start;
	/**
	 * Stop value.
	 */
	protected final int stop;
	/**
	 * Duration.
	 */
	protected final long duration;

	/**
	 * Start time.
	 *
	 * @see #start()
	 */
	protected long startTime;
	/**
	 * {@code true} if finished, {@code false} otherwise.
	 *
	 * @see #isFinished()
	 * @see #isFinished(int)
	 */
	protected boolean finished;

	/**
	 * Creates a motion for a move from start to stop.
	 *
	 * @param start
	 *            the start value.
	 * @param stop
	 *            the stop value.
	 * @param duration
	 *            the maximum duration of the motion.
	 */
	public AbstractMotion(int start, int stop, long duration) {
		this.start = start;
		this.stop = stop;
		this.duration = duration;
		start(); // start at creation by default
	}

	@Override
	public long getDuration() {
		return this.duration;
	}

	@Override
	public void start() {
		this.finished = false;
		this.startTime = Util.platformTimeMillis();
	}

	@Override
	public boolean isFinished() {
		long time = Util.platformTimeMillis();
		return this.finished || time >= this.startTime + this.duration;
	}

	@Override
	public int getStartValue() {
		return this.start;
	}

	@Override
	public int getStopValue() {
		return this.stop;
	}

	@Override
	public int getCurrentValue() {
		if (!isFinished()) {
			long time = Util.platformTimeMillis();
			long elapsed = (time - this.startTime);
			int value = computeCurrentValue(elapsed);
			if (!isFinished(value)) {
				return value;
			}
			this.finished = true;
		}
		return this.stop;
	}

	@Override
	public int getValue(long elapsed) {
		if (elapsed <= 0) {
			return this.start;
		} else if (elapsed >= this.duration) {
			return this.stop;
		} else {
			return computeCurrentValue(elapsed);
		}
	}

	/**
	 * Checks whether the motion is finished or not, considering the given value.
	 * <p>
	 * By default, the motion is finished if the value reach (or exceed) the stop value.
	 *
	 * @param value
	 *            the value to check with.
	 * @return {@code true} if the motion is finished, {@code false} otherwise.
	 */
	protected boolean isFinished(int value) {
		return (this.stop - this.start > 0) ? (value >= this.stop) : (value <= this.stop);
	}

	/**
	 * Computes current value.
	 *
	 * @param elapsed
	 *            the elapsed time since the beginning.
	 * @return the current value.
	 */
	protected abstract int computeCurrentValue(long elapsed);

}
