/*
 * Java
 *
 * Copyright 2018-2024 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.bluetooth.util.services.cts;

import ej.bluetooth.BluetoothProperties;
import ej.bluetooth.BluetoothServiceDefinition;
import ej.bluetooth.BluetoothUuid;
import ej.bluetooth.util.ServiceHelper;

/**
 * Provides constants and static methods related to the standard current time service.
 */
public class CurrentTimeConstants {

	/** The UUID of the current time service. */
	public static final BluetoothUuid SERVICE_UUID = new BluetoothUuid(0x1805);

	/** The UUID of the current time characteristic. */
	public static final BluetoothUuid CURRENT_TIME_UUID = new BluetoothUuid(0x2A2B);

	/** The UUID of the local time info characteristic. */
	public static final BluetoothUuid LOCAL_TIME_INFO_UUID = new BluetoothUuid(0x2A0F);

	private CurrentTimeConstants() {
		// private constructor
	}

	/**
	 * Gets the definition of the current time service.
	 *
	 * @param securityLevel
	 *            the security level (see {@link ej.bluetooth.util.SecurityLevels}).
	 * @return the definition of the current time service.
	 */
	public static BluetoothServiceDefinition getServiceDefinition(int securityLevel) {
		int readPermission = ServiceHelper.getReadPermission(securityLevel);

		BluetoothServiceDefinition service = new BluetoothServiceDefinition(SERVICE_UUID);
		service.addCharacteristic(CURRENT_TIME_UUID, BluetoothProperties.READ | BluetoothProperties.NOTIFY,
				readPermission);
		service.addCharacteristic(LOCAL_TIME_INFO_UUID, BluetoothProperties.READ, readPermission);
		return service;
	}
}
