package java.lang;

import java.security.Permission;

/**
 * The security manager is a class that allows applications to implement a security policy. It
 * allows an application to determine, before performing a possibly unsafe or sensitive operation,
 * what the operation is and whether it is being attempted in a security context that allows the
 * operation to be performed. The application can allow or disallow the operation.
 * <p>
 * The <code>SecurityManager</code> class contains many methods with names that begin with the word
 * <code>check</code>. These methods are called by various methods in the Java libraries before
 * those methods perform certain potentially sensitive operations. The invocation of such a
 * <code>check</code> method typically looks like this:
 * <blockquote>
 *
 * <pre>
 *     SecurityManager security = System.getSecurityManager();
 *     if (security != null) {
 *         security.check<i>XXX</i>(argument, &nbsp;.&nbsp;.&nbsp;.&nbsp;);
 *     }
 * </pre>
 *
 * </blockquote>
 * <p>
 * The security manager is thereby given an opportunity to prevent completion of the operation by
 * throwing an exception. A security manager routine simply returns if the operation is permitted,
 * but throws a <code>SecurityException</code> if the operation is not permitted. The only exception
 * to this convention is <code>checkTopLevelWindow</code>, which returns a <code>boolean</code>
 * value.
 * <p>
 * The current security manager is set by the <code>setSecurityManager</code> method in class
 * <code>System</code>. The current security manager is obtained by the
 * <code>getSecurityManager</code> method.
 * <p>
 * The special method {@link SecurityManager#checkPermission(java.security.Permission)} determines
 * whether an access request indicated by a specified permission should be granted or denied.
 *
 * <p>
 * If a requested access is allowed, <code>checkPermission</code> returns quietly. If denied, a
 * <code>SecurityException</code> is thrown.
 * <p>
 * As of Java 2 SDK v1.2, the default implementation of each of the other <code>check</code> methods
 * in <code>SecurityManager</code> is to call the <code>SecurityManager checkPermission</code>
 * method to determine if the calling thread has permission to perform the requested operation.
 * <p>
 * Note that the <code>checkPermission</code> method with just a single permission argument always
 * performs security checks within the context of the currently executing thread.
 *
 * <p>
 * Permissions fall into these categories: Runtime and Property. The classes managing these various
 * permission categories are <code>java.lang.RuntimePermission</code>,
 * <code>java.util.PropertyPermission</code>,
 *
 * <p>
 * Some of the permission classes have an "actions" list that tells the actions that are permitted
 * for the object.
 *
 * <p>
 * Other permission classes are for "named" permissions - ones that contain a name but no actions
 * list; you either have the named permission or you don't.
 *
 * <p>
 * See <a href ="../../../technotes/guides/security/permissions.html"> Permissions in the JDK</a>
 * for permission-related information. This document includes, for example, a table listing the
 * various SecurityManager <code>check</code> methods and the permission(s) the default
 * implementation of each such method requires. It also contains a table of all the version 1.2
 * methods that require permissions, and for each such method tells which permission it requires.
 * <p>
 * For more information about <code>SecurityManager</code> changes made in the JDK and advice
 * regarding porting of 1.1-style security managers, see the
 * <a href="../../../technotes/guides/security/index.html">security documentation</a>.
 *
 * @see java.lang.SecurityException
 * @see java.lang.System#getSecurityManager() getSecurityManager
 * @see java.lang.System#setSecurityManager(java.lang.SecurityManager) setSecurityManager
 * @see java.security.Permission
 * @see java.security.BasicPermission
 */
public class SecurityManager {

	/**
	 * Constructs a new <code>SecurityManager</code>.
	 *
	 * <p>
	 * If there is a security manager already installed, this method first calls the security manager's
	 * <code>checkPermission</code> method with the
	 * <code>RuntimePermission("createSecurityManager")</code> permission to ensure the calling thread
	 * has permission to create a new security manager. This may result in throwing a
	 * <code>SecurityException</code>.
	 *
	 * @exception java.lang.SecurityException
	 *            if a security manager already exists and its <code>checkPermission</code> method
	 *            doesn't allow creation of a new security manager.
	 * @see java.lang.System#getSecurityManager()
	 * @see #checkPermission(java.security.Permission) checkPermission
	 * @see java.lang.RuntimePermission
	 */
	public SecurityManager() {
		throw new RuntimeException();
	}

	/**
	 * Throws a <code>SecurityException</code> if the calling thread is not allowed to modify the thread
	 * argument.
	 * <p>
	 * This method is invoked for the current security manager by the <code>setPriority</code> and
	 * <code>setName</code> methods of class <code>Thread</code>.
	 * <p>
	 * If the thread argument is a system thread then this method calls <code>checkPermission</code>
	 * with the <code>RuntimePermission("modifyThread")</code> permission. If the thread argument is
	 * <i>not</i> a system thread, this method just returns silently.
	 * <p>
	 * Applications that want a stricter policy should override this method. If this method is
	 * overridden, the method that overrides it should additionally check to see if the calling thread
	 * has the <code>RuntimePermission("modifyThread")</code> permission, and if so, return silently.
	 * This is to ensure that code granted that permission is allowed to manipulate any thread.
	 * <p>
	 * If this method is overridden, then <code>super.checkAccess</code> should be called by the first
	 * statement in the overridden method, or the equivalent security check should be placed in the
	 * overridden method.
	 *
	 * @param t
	 *        the thread to be checked.
	 * @exception SecurityException
	 *            if the calling thread does not have permission to modify the thread.
	 * @exception NullPointerException
	 *            if the thread argument is <code>null</code>.
	 * @see java.lang.Thread#setName(java.lang.String) setName
	 * @see java.lang.Thread#setPriority(int) setPriority
	 * @see #checkPermission(java.security.Permission) checkPermission
	 */
	public void checkAccess(Thread t) {
		throw new RuntimeException();
	}

	/**
	 * Throws a <code>SecurityException</code> if the requested access, specified by the given
	 * permission, is not permitted based on the security policy currently in effect.
	 * <p>
	 *
	 * @param perm
	 *        the requested permission.
	 * @exception SecurityException
	 *            if access is not permitted based on the current security policy.
	 * @exception NullPointerException
	 *            if the permission argument is <code>null</code>.
	 */
	public void checkPermission(Permission perm) {
		throw new RuntimeException();
	}

}
