/*
 * Copyright (c) 1997, 2012, Oracle and/or its affiliates. All rights reserved.
 * Copyright (C) 2015-2020 MicroEJ Corp. - EDC compliance and optimizations.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package javax.net.ssl;

import java.io.IOException;
import java.io.InputStream;
import java.net.Socket;

import javax.net.SocketFactory;

import ej.annotation.Nullable;

/**
 * <code>SSLSocketFactory</code>s create <code>SSLSocket</code>s.
 *
 * @since 1.4
 * @see SSLSocket
 * @author David Brownell
 */
public abstract class SSLSocketFactory extends SocketFactory {
	/**
	 * Constructor is used only by subclasses.
	 */
	public SSLSocketFactory() {
		throw new RuntimeException();
	}

	/**
	 * Returns the default SSL socket factory.
	 *
	 * <p>
	 * The first time this method is called, the security property "ssl.SocketFactory.provider" is examined. If it is
	 * non-null, a class by that name is loaded and instantiated. If that is successful and the object is an instance of
	 * SSLSocketFactory, it is made the default SSL socket factory.
	 *
	 * <p>
	 * Otherwise, this method returns <code>SSLContext.getDefault().getSocketFactory()</code>. If that call fails, an
	 * inoperative factory is returned.
	 *
	 * @return the default <code>SocketFactory</code>
	 * @see SSLContext#getDefault
	 */
	public static synchronized SocketFactory getDefault() {
		throw new RuntimeException();
	}

	/**
	 * Returns a socket layered over an existing socket connected to the named host, at the given port. This constructor
	 * can be used when tunneling SSL through a proxy or when negotiating the use of SSL over an existing socket. The
	 * host and port refer to the logical peer destination. This socket is configured using the socket options
	 * established for this factory.
	 *
	 * @param s
	 *            the existing socket
	 * @param host
	 *            the server host
	 * @param port
	 *            the server port
	 * @param autoClose
	 *            close the underlying socket when this socket is closed
	 * @return a socket connected to the specified host and port
	 * @throws IOException
	 *             if an I/O error occurs when creating the socket
	 * @throws NullPointerException
	 *             if the parameter s is null
	 */
	public abstract Socket createSocket(Socket s, String host, int port, boolean autoClose) throws IOException;

	/**
	 * Creates a server mode {@link Socket} layered over an existing connected socket, and is able to read data which
	 * has already been consumed/removed from the {@link Socket}'s underlying {@link InputStream}.
	 * <p>
	 * This method can be used by a server application that needs to observe the inbound data but still create valid
	 * SSL/TLS connections: for example, inspection of Server Name Indication (SNI) extensions (See section 3 of
	 * <A HREF="http://www.ietf.org/rfc/rfc6066.txt">TLS Extensions (RFC6066)</A>). Data that has been already removed
	 * from the underlying {@link InputStream} should be loaded into the {@code consumed} stream before this method is
	 * called, perhaps using a {@link java.io.ByteArrayInputStream}. When this {@link Socket} begins handshaking, it
	 * will read all of the data in {@code consumed} until it reaches {@code EOF}, then all further data is read from
	 * the underlying {@link InputStream} as usual.
	 * <p>
	 * The returned socket is configured using the socket options established for this factory, and is set to use server
	 * mode when handshaking.
	 *
	 * @param s
	 *            the existing socket
	 * @param consumed
	 *            the consumed inbound network data that has already been removed from the existing {@link Socket}
	 *            {@link InputStream}. This parameter may be {@code null} if no data has been removed.
	 * @param autoClose
	 *            close the underlying socket when this socket is closed.
	 *
	 * @return the {@link Socket} compliant with the socket options established for this factory
	 *
	 * @throws IOException
	 *             if an I/O error occurs when creating the socket
	 * @throws UnsupportedOperationException
	 *             if the underlying provider does not implement the operation
	 * @throws NullPointerException
	 *             if {@code s} is {@code null}
	 *
	 * @since 1.8
	 */
	public Socket createSocket(Socket s, @Nullable InputStream consumed, boolean autoClose) throws IOException {
		throw new UnsupportedOperationException();
	}
}