/*
 * Java
 *
 * Copyright 2024 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microvg;

import java.awt.Composite;
import java.awt.Graphics2D;
import java.awt.geom.AffineTransform;
import java.awt.geom.GeneralPath;
import java.awt.geom.Path2D;
import java.awt.geom.Rectangle2D;
import java.util.ServiceLoader;

import ej.microui.display.LLUIDisplay;
import ej.microui.display.MicroUIGraphicsContext;
import ej.microui.display.MicroUIImageFormat;
import ej.microvg.bvi.BufferedVectorImageDrawer;
import ej.microvg.paint.VGPaint;

/**
 * Equivalent of {@link LLUIDisplay} for MicroVG: it provides some methods to map MicroVG byte arrays in MicroVG
 * Graphics Context objects, to manipulate MicroVG linear gradient, etc.
 * <p>
 * An instance of this framework is available via the {@link #Instance} field. To use the framework methods, caller must
 * synchronize its calls on this {@link #Instance} field.
 */
public interface LLVGEngine {

	/**
	 * Buffered Vector Image RAW format (same format is used in application library).
	 */
	MicroUIImageFormat BUFFERED_VECTOR_IMAGE_FORMAT = MicroUIImageFormat.MICROUI_IMAGE_FORMAT_CUSTOM_7;

	/**
	 * Unique instance of MicroVG Engine. It provides some methods to to map MicroVG byte arrays in MicroVG Graphics
	 * Context objects, to manipulate MicroVG linear gradient etc.
	 * <p>
	 * To use the framework methods, caller must synchronize its calls on this {@link #Instance} field.
	 */
	LLVGEngine Instance = ServiceLoader.load(LLVGEngine.class, LLVGEngine.class.getClassLoader()).iterator().next();

	/**
	 * Registers a drawer that supports a specific format.
	 *
	 * @param drawer
	 *            the drawer
	 * @see VGDrawing#handledFormat()
	 */
	void registerVGDrawer(VGDrawing drawer);

	/**
	 * Unregisters the drawer registered for the given format.
	 *
	 * @param format
	 *            the now unsupported format
	 * @see VGDrawing#handledFormat()
	 */
	void unregisterVGDrawer(MicroUIImageFormat format);

	/**
	 * Gets the drawer used to draw on the given graphics context.
	 *
	 * @param gc
	 *            the graphics context to draw on
	 * @return the drawer registered for the format of the graphics context or a stub if none
	 */
	VGDrawing getVGDrawer(MicroUIGraphicsContext gc);

	/**
	 * Gets the drawer used to draw with the given format.
	 *
	 * @param format
	 *            the format
	 * @return the drawer registered for the format or a stub if none
	 */
	VGDrawing getVGDrawer(MicroUIImageFormat format);

	/**
	 * Maps an AWT general path on the MicroVG path array.
	 *
	 * @param path
	 *            the MicroVG path
	 * @return an AWT path
	 */
	GeneralPath mapPath(byte[] path);

	/**
	 * Maps a {@link VGPaint} from the gradient's array, applying the gradient's transformation, the drawing's
	 * transformation and the drawing's opacity.
	 *
	 * @param gradient
	 *            the gradient's array
	 * @param gradientMatrix
	 *            the gradient's transformation
	 * @param globalMatrix
	 *            the drawing's transformation
	 * @param globalAlpha
	 *            the drawing's opacity
	 * @return an AWT paint object
	 */
	VGPaint mapGradient(int[] gradient, float[] gradientMatrix, float[] globalMatrix, int globalAlpha);

	/**
	 * Gets the unique image identifier from the SNI context.
	 *
	 * @param sniContext
	 *            the representation of the MicroVG image
	 * @return the expected raw image or {@code null} if the image has been closed
	 */
	MicroVGImage mapVectorImage(byte[] sniContext);

	/**
	 * Sets a clip; cannot be higher than master clip.
	 *
	 * @param g2d
	 *            Graphics2D drawing object
	 * @param gc
	 *            MicroUI GraphicsContext where performing the drawing
	 * @param path
	 *            the path to clip.
	 * @param at
	 *            the path transformation
	 */
	void setClip(Graphics2D g2d, MicroUIGraphicsContext gc, GeneralPath path, AffineTransform at);

	/**
	 * Gets the AWT fill rule according to the MicroVG fill rule.
	 *
	 * @param fillRule
	 *            a MicroVG fill rule (example: {@link LLVGConstants#FILLTYPE_EVEN_ODD}
	 * @return a AWT fill rule (example: {@link Path2D#WIND_EVEN_ODD}
	 */
	int getFillRule(int fillRule);

	/**
	 * Gets the AWT composite according to the MicroVG blend mode.
	 *
	 * @param blend
	 *            a MicroVG blend mode (example: {@link LLVGConstants#BLEND_SRC}
	 * @return a AWT composite that fits the MicroVG blend mode specification
	 */
	Composite getComposite(int blend);

	/**
	 * Decomposes a string in glyphs and draws them using the drawer for the given graphics context
	 * ({@link VGDrawing#drawPath(MicroUIGraphicsContext, GeneralPath, float, float, float[], int, int, VGPaint)}.
	 *
	 * @param gc
	 *            the destination
	 * @param string
	 *            the array of characters to draw
	 * @param face
	 *            the font face to use
	 * @param size
	 *            the font size, in pixels
	 * @param x
	 *            the destination X coordinate
	 * @param y
	 *            the destination Y coordinate
	 * @param matrix
	 *            the deformation to apply
	 * @param blendMode
	 *            the blend mode
	 * @param p
	 *            the color representation to use
	 * @param letterSpacing
	 *            the extra letter spacing to use
	 * @param radius
	 *            the radius of the circle
	 * @param direction
	 *            the direction of the text along the circle
	 * @return {@link LLVGPainter#RETURN_SUCCESS} on success or {@link LLVGPainter#RESOURCE_CLOSED} if the font is
	 *         closed
	 */
	int drawString(MicroUIGraphicsContext gc, char[] string, int face, float size, float x, float y, float[] matrix,
			int blendMode, VGPaint p, float letterSpacing, float radius, int direction);

	/**
	 * Truncates the pixels' colors of the given region to take in consideration the hardware display pixel
	 * representation constraints.
	 *
	 * @param gc
	 *            the target to update
	 * @param bounds
	 *            the region to update
	 */
	default void convertColorToColorToDraw(MicroUIGraphicsContext gc, Rectangle2D bounds) {
		LLUIDisplay d = LLUIDisplay.Instance;
		d.convertRegion(gc, (int) bounds.getMinX(), (int) bounds.getMinY(), (int) bounds.getWidth(),
				(int) bounds.getHeight());
	}

	/**
	 * Gets the buffered vector image drawer that can be used to draw in a buffered vector image.
	 *
	 * @return the buffered vector image drawer
	 */
	BufferedVectorImageDrawer getBufferedVectorImageDrawer();

}
