/*
 * Java
 *
 * Copyright 2023 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.event;

import java.io.IOException;


/**
 * Native methods of the Event Queue Foundation Library.
 */
public class EventQueueNatives {

	/**
	 * Private constructor.
	 */
	private EventQueueNatives() {

	}

	/**
	 * Initializes the event queue.
	 */
	public static native void initialize();

	/**
	 * Offers an event to the queue.
	 *
	 * @param type the type of the event.
	 * @param data the data of the event.
	 * @return true if the message has been sent, false otherwise.
	 */
	public static native boolean offerEvent(int type, int data);

	/**
	 * Offers an extended event to the queue.
	 *
	 * @param type   the type of the event.
	 * @param data   the data of the event.
	 * @param length the number of bytes in the data.
	 * @return true if the message has been sent, false otherwise.
	 */
	public static native boolean offerExtendedEvent(int type, byte[] data, int length);

	/**
	 * Waits for an event from the queue and returns it.
	 *
	 * @return the read event.
	 */
	public static native int waitEvent();

	// EventQueueDataReader native methods.

	/**
	 * Starts to read an extended data.
	 *
	 * @param dataLength the number of bytes of the extended data.
	 */
	public static native void startReadExtendedData(int dataLength);

	/**
	 * The Java listener finished to read the data from the event queue. If there is
	 * any left data left in the queue, purge it.
	 */
	public static native void endReadExtendedData();

	/**
	 * Gets the next boolean of data.
	 *
	 * @return the boolean value.
	 * @throws IOException if there is no boolean remaining in the data of the
	 *                     extended event.
	 */
	public static native boolean nativeReadBoolean() throws IOException;

	/**
	 * Gets the next byte of data.
	 *
	 * @return the byte value.
	 * @throws IOException if there is no byte remaining in the data of the extended
	 *                     event.
	 */
	public static native byte nativeReadByte() throws IOException;

	/**
	 * Gets the next char of data.
	 *
	 * @return the char value.
	 * @throws IOException if there is no char remaining in the data of the extended
	 *                     event.
	 */
	public static native char nativeReadChar() throws IOException;

	/**
	 * Gets the next double of data.
	 *
	 * @return the double value.
	 * @throws IOException if there is no double remaining in the data of the
	 *                     extended event.
	 */
	public static native double nativeReadDouble() throws IOException;

	/**
	 * Gets the next float of data.
	 *
	 * @return the float value.
	 * @throws IOException if there is no float remaining in the data of the
	 *                     extended event.
	 */
	public static native float nativeReadFloat() throws IOException;

	/**
	 * Reads up to len bytes of data and store them into the buffer array b.
	 *
	 * @param b   the buffer into which the data is read.
	 * @param off the start offset in array b at which the data is written.
	 * @param len the number of bytes to read.
	 * @return the total number of bytes read into the buffer.
	 * @throws IOException if there are not enough bytes remaining in the data of
	 *                     the extended event or if the buffer is too small to store
	 *                     the event data.
	 */
	public static native int nativeRead(byte[] b, int off, int len) throws IOException;

	/**
	 * Gets the next integer of data.
	 *
	 * @return the integer value.
	 * @throws IOException if there is no integer remaining in the data of the
	 *                     extended event.
	 */
	public static native int nativeReadInt() throws IOException;

	/**
	 * Gets the next long of data.
	 *
	 * @return the long value.
	 * @throws IOException if there is no long remaining in the data of the extended
	 *                     event.
	 */
	public static native long nativeReadLong() throws IOException;

	/**
	 * Gets the next short of data.
	 *
	 * @return the short value.
	 * @throws IOException if there is no short remaining in the data of the
	 *                     extended event.
	 */
	public static native short nativeReadShort() throws IOException;

	/**
	 * Gets the next unsigned byte of data.
	 *
	 * @return the unsigned byte value.
	 * @throws IOException if there is no unsigned byte remaining in the data of the
	 *                     extended event.
	 */
	public static native int nativeReadUnsignedByte() throws IOException;

	/**
	 * Gets the next unsigned short of data.
	 *
	 * @return the unsigned short value.
	 * @throws IOException if there is no unsigned short remaining in the data of
	 *                     the extended event.
	 */
	public static native int nativeReadUnsignedShort() throws IOException;

	/**
	 * Skips n bytes.
	 *
	 * @param n the number of bytes to skip.
	 * @return 0 if OK and -1 if an error occurs.
	 */
	public static native int nativeSkipBytes(int n);

	/**
	 * Returns the number of bytes that can be read (or skipped over).
	 *
	 * @return the number of bytes available in the remaining extended data.
	 */
	public static native int nativeAvailable();
}
