/*
 * <Java>
 *
 * Copyright 2013-2023 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.service.loader;

import ej.annotation.Nullable;

/**
 * Service loader implementation that creates service instance from a declared implementation class name.
 * <p>
 * Subclasses need to provide the implementation class name matching the requested services. The implementation is then
 * instantiated using {@link Class#forName(String)} and {@link Class#newInstance()}. The singletons are stored in a map.
 */
public abstract class DependencyInjectionServiceLoader extends SimpleServiceLoader {

	@Override
	@Nullable
	protected <T> T createAlternativeImplementation(Class<T> service) {
		String serviceName = service.getName();
		String implementationName = getImplementationName(serviceName);
		if (implementationName != null) {
			T serviceImplementation = ServiceLoaderHelper.createClassInstance(service, implementationName);
			putService(service, serviceImplementation);
			return serviceImplementation;
		}
		return null;
	}

	/**
	 * Gets the implementation class name for the given service name.
	 *
	 * @param serviceName
	 *            the service name.
	 * @return the implementation class name, or {@code null} if none.
	 */
	@Nullable
	protected abstract String getImplementationName(String serviceName);

}