/*
 * Java
 *
 * Copyright 2020 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.drawing;

import ej.drawing.LLDWPainter.DrawingCap;
import ej.drawing.LLDWPainter.DrawingFlip;
import ej.microui.display.LLUIDisplay;
import ej.microui.display.LLUIPainter.MicroUIGraphicsContext;
import ej.microui.display.LLUIPainter.MicroUIImage;

/**
 * This interface provides all drawing methods useful to implement ej.drawing library drawing native methods.
 * <p>
 * This interface is implemented by the MicroUI graphical engine's internal software drawer. This drawer implements all
 * drawings in software. The rendering is exactly the same than the embedded algorithms listed in
 * <code>dw_drawing_soft.h</code>. See {@link LLUIDisplay#getDWDrawerSoftware()}.
 * <p>
 * This interface can be fully implemented by a dedicated platform class in platform front panel project. This allows to
 * implement drawing algorithms like embedded platform.
 * <p>
 * Additionally, this interface can be only partially implemented by a dedicated platform class in platform front panel
 * project. This class must implement {@link DWDrawingDefault} interface instead and override only a set of drawing
 * methods. This allows to implement same drawing algorithms like embedded platform. All others drawing algorithms are
 * using the front panel's internal software drawer.
 */
public interface DWDrawing {

	/**
	 * Default implementation of {@link DWDrawing} which uses the getUIDrawerSoftware's internal software drawer to
	 * perform all drawings. This interface can be implemented to override only a set of drawing methods which matches
	 * with the embedded implementation.
	 */
	public interface DWDrawingDefault extends DWDrawing {

		@Override
		default void drawThickFadedPoint(MicroUIGraphicsContext gc, int x, int y, int thickness, int fade) {
			LLUIDisplay.Instance.getDWDrawerSoftware().drawThickFadedPoint(gc, x, y, thickness, fade);
		}

		@Override
		default void drawThickFadedLine(MicroUIGraphicsContext gc, int x1, int y1, int x2, int y2, int thickness,
				int fade, DrawingCap start, DrawingCap end) {
			LLUIDisplay.Instance.getDWDrawerSoftware().drawThickFadedLine(gc, x1, y1, x2, y2, thickness, fade, start,
					end);
		}

		@Override
		default void drawThickFadedCircle(MicroUIGraphicsContext gc, int x, int y, int diameter, int thickness, int fade) {
			LLUIDisplay.Instance.getDWDrawerSoftware().drawThickFadedCircle(gc, x, y, diameter, thickness, fade);
		}

		@Override
		default void drawThickFadedCircleArc(MicroUIGraphicsContext gc, int x, int y, int diameter, float startAngle,
				float arcAngle, int thickness, int fade, DrawingCap start, DrawingCap end) {
			LLUIDisplay.Instance.getDWDrawerSoftware().drawThickFadedCircleArc(gc, x, y, diameter, startAngle, arcAngle,
					thickness, fade, start, end);
		}

		@Override
		default void drawThickFadedEllipse(MicroUIGraphicsContext gc, int x, int y, int width, int height, int thickness,
				int fade) {
			LLUIDisplay.Instance.getDWDrawerSoftware().drawThickFadedEllipse(gc, x, y, width, height, thickness, fade);
		}

		@Override
		default void drawThickLine(MicroUIGraphicsContext gc, int x1, int y1, int x2, int y2, int thickness) {
			LLUIDisplay.Instance.getDWDrawerSoftware().drawThickLine(gc, x1, y1, x2, y2, thickness);
		}

		@Override
		default void drawThickCircle(MicroUIGraphicsContext gc, int x, int y, int diameter, int thickness) {
			LLUIDisplay.Instance.getDWDrawerSoftware().drawThickCircle(gc, x, y, diameter, thickness);
		}

		@Override
		default void drawThickEllipse(MicroUIGraphicsContext gc, int x, int y, int width, int height, int thickness) {
			LLUIDisplay.Instance.getDWDrawerSoftware().drawThickEllipse(gc, x, y, width, height, thickness);
		}

		@Override
		default void drawThickCircleArc(MicroUIGraphicsContext gc, int x, int y, int diameter, float startAngle,
				float arcAngle, int thickness) {
			LLUIDisplay.Instance.getDWDrawerSoftware().drawThickCircleArc(gc, x, y, diameter, startAngle, arcAngle,
					thickness);
		}

		@Override
		default void drawFlippedImage(MicroUIGraphicsContext gc, MicroUIImage img, int regionX, int regionY, int width,
				int height, int x, int y, DrawingFlip flip, int alpha) {
			LLUIDisplay.Instance.getDWDrawerSoftware().drawFlippedImage(gc, img, regionX, regionY, width, height, x, y, flip,
					alpha);
		}

		@Override
		default void drawRotatedImageNearestNeighbor(MicroUIGraphicsContext gc, MicroUIImage img, int x, int y,
				int rotationX, int rotationY, float angle, int alpha) {
			LLUIDisplay.Instance.getDWDrawerSoftware().drawRotatedImageNearestNeighbor(gc, img, x, y, rotationX, rotationY,
					angle, alpha);
		}

		@Override
		default void drawRotatedImageBilinear(MicroUIGraphicsContext gc, MicroUIImage img, int x, int y, int rotationX,
				int rotationY, float angle, int alpha) {
			LLUIDisplay.Instance.getDWDrawerSoftware().drawRotatedImageBilinear(gc, img, x, y, rotationX, rotationY, angle,
					alpha);
		}

		@Override
		default void drawScaledImageNearestNeighbor(MicroUIGraphicsContext gc, MicroUIImage img, int x, int y, float factorX,
				float factorY, int alpha) {
			LLUIDisplay.Instance.getDWDrawerSoftware().drawScaledImageNearestNeighbor(gc, img, x, y, factorX, factorY,
					alpha);
		}

		@Override
		default void drawScaledImageBilinear(MicroUIGraphicsContext gc, MicroUIImage img, int x, int y, float factorX,
				float factorY, int alpha) {
			LLUIDisplay.Instance.getDWDrawerSoftware().drawScaledImageBilinear(gc, img, x, y, factorX, factorY, alpha);
		}
	}

	/**
	 * Draws a thick point with fade at given position.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the point X coordinate.
	 * @param y
	 *            the point Y coordinate.
	 * @param thickness
	 *            the point thickness.
	 * @param fade
	 *            the fade to apply.
	 */
	void drawThickFadedPoint(MicroUIGraphicsContext gc, int x, int y, int thickness, int fade);

	/**
	 * Draws a thick line with fade between given points.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param x1
	 *            the x coordinate of the start of the line
	 * @param y1
	 *            the y coordinate of the start of the line
	 * @param x2
	 *            the x coordinate of the end of the line
	 * @param y2
	 *            the y coordinate of the end of the line
	 * @param thickness
	 *            the line thickness.
	 * @param fade
	 *            the fade to apply.
	 * @param start
	 *            cap representation of start of shape
	 * @param end
	 *            cap representation of end of shape
	 */
	void drawThickFadedLine(MicroUIGraphicsContext gc, int x1, int y1, int x2, int y2, int thickness, int fade,
			DrawingCap start, DrawingCap end);

	/**
	 * Draws a thick circle with fade covering the square specified by its diameter. If diameter is negative or zero,
	 * nothing is drawn.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the x coordinate of the upper-left corner of the square where the circle is drawn
	 * @param y
	 *            the y coordinate of the upper-left corner of the square where the circle is drawn
	 * @param diameter
	 *            the diameter of the circle to draw
	 * @param thickness
	 *            the circle thickness.
	 * @param fade
	 *            the fade to apply.
	 */
	void drawThickFadedCircle(MicroUIGraphicsContext gc, int x, int y, int diameter, int thickness, int fade);

	/**
	 * Draws a thick circle with fade arc covering the specified square. The arc is drawn from startAngle up to arcAngle
	 * degrees.
	 * <p>
	 * The center of the arc is defined as the center of the square whose origin is at (x,y) (upper-left corner) and
	 * whose dimension is given by diameter.
	 * <p>
	 * Angles are interpreted such that 0 degrees is at the 3 o'clock position. A positive value indicates a
	 * counter-clockwise rotation while a negative value indicates a clockwise rotation.
	 * <p>
	 * If diameter is negative or zero, nothing is drawn.
	 * <p>
	 * The angles are given relative to the square. For instance an angle of 45 degrees is always defined by the line
	 * from the center of the square to the upper right corner of the square. Thus for a non squared square angles are
	 * skewed along either height or width.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the x coordinate of the upper-left corner of the square where the arc is drawn
	 * @param y
	 *            the y coordinate of the upper-left corner of the square where the arc is drawn
	 * @param diameter
	 *            the diameter of the circle to draw
	 * @param startAngle
	 *            the beginning angle of the arc to draw
	 * @param arcAngle
	 *            the angular extent of the arc from startAngle
	 * @param thickness
	 *            the arc thickness.
	 * @param fade
	 *            the fade to apply.
	 * @param start
	 *            cap representation of start of shape
	 * @param end
	 *            cap representation of end of shape
	 */
	void drawThickFadedCircleArc(MicroUIGraphicsContext gc, int x, int y, int diameter, float startAngle, float arcAngle,
			int thickness, int fade, DrawingCap start, DrawingCap end);

	/**
	 * Draws a thick ellipse with fade covering the specified rectangle.
	 * <p>
	 * The center of the ellipse is defined as the center of the rectangle whose origin is at (x,y) (upper-left corner)
	 * and whose dimension is given by width and height.
	 * <p>
	 * If either width or height is negative or zero, nothing is drawn.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the x coordinate of the upper-left corner of the rectangle where the ellipse is drawn
	 * @param y
	 *            the y coordinate of the upper-left corner of the rectangle where the ellipse is drawn
	 * @param width
	 *            the width of the ellipse to draw
	 * @param height
	 *            the height of the ellipse to draw
	 * @param thickness
	 *            the ellipse thickness.
	 * @param fade
	 *            the fade to apply.
	 */
	void drawThickFadedEllipse(MicroUIGraphicsContext gc, int x, int y, int width, int height, int thickness, int fade);

	/**
	 * Draws a thick line between given points.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param x1
	 *            the x coordinate of the start of the line
	 * @param y1
	 *            the y coordinate of the start of the line
	 * @param x2
	 *            the x coordinate of the end of the line
	 * @param y2
	 *            the y coordinate of the end of the line
	 * @param thickness
	 *            the line thickness.
	 */
	void drawThickLine(MicroUIGraphicsContext gc, int x1, int y1, int x2, int y2, int thickness);

	/**
	 * Draws a thick circle covering the square specified by its diameter.
	 * <p>
	 * If diameter is negative or zero, nothing is drawn.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the x coordinate of the upper-left corner of the square where the circle is drawn
	 * @param y
	 *            the y coordinate of the upper-left corner of the square where the circle is drawn
	 * @param diameter
	 *            the diameter of the circle to draw
	 * @param thickness
	 *            the circle thickness.
	 */
	void drawThickCircle(MicroUIGraphicsContext gc, int x, int y, int diameter, int thickness);

	/**
	 * Draws a thick ellipse covering the specified rectangle.
	 * <p>
	 * The center of the ellipse is defined as the center of the rectangle whose origin is at (x,y) (upper-left corner)
	 * and whose dimension is given by width and height.
	 * <p>
	 * If either width or height is negative or zero, nothing is drawn.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the x coordinate of the upper-left corner of the rectangle where the ellipse is drawn
	 * @param y
	 *            the y coordinate of the upper-left corner of the rectangle where the ellipse is drawn
	 * @param width
	 *            the width of the ellipse to draw
	 * @param height
	 *            the height of the ellipse to draw
	 * @param thickness
	 *            the ellipse thickness.
	 */
	void drawThickEllipse(MicroUIGraphicsContext gc, int x, int y, int width, int height, int thickness);

	/**
	 * Draws a thick arc covering the square specified by its diameter.
	 * <p>
	 * The arc is drawn from startAngle up to arcAngle degrees. The center of the arc is defined as the center of the
	 * square whose origin is at (x,y) (upper-left corner) and whose dimension is given by diameter.
	 * <p>
	 * Angles are interpreted such that 0 degrees is at the 3 o'clock position. A positive value indicates a
	 * counter-clockwise rotation while a negative value indicates a clockwise rotation.
	 * <p>
	 * If diameter is negative, nothing is drawn.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the x coordinate of the upper-left corner of the square where the arc is drawn
	 * @param y
	 *            the y coordinate of the upper-left corner of the square where the arc is drawn
	 * @param diameter
	 *            the diameter of the circle to draw
	 * @param startAngle
	 *            the beginning angle of the arc to draw
	 * @param arcAngle
	 *            the angular extent of the arc from startAngle
	 * @param thickness
	 *            the arc thickness.
	 */
	void drawThickCircleArc(MicroUIGraphicsContext gc, int x, int y, int diameter, float startAngle, float arcAngle,
			int thickness);

	/**
	 * Draws an image applying a flip (0, 90, 180 or 270 degrees with or without mirror).
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param img
	 *            the MicroUI Image to draw.
	 * @param regionX
	 *            the x coordinate of the upper-left corner of the region to draw.
	 * @param regionY
	 *            the x coordinate of the upper-left corner of the region to draw.
	 * @param width
	 *            the width of the region to copy.
	 * @param height
	 *            the height of the region to copy.
	 * @param x
	 *            the x coordinate of the top-left point in the destination.
	 * @param y
	 *            the y coordinate of the top-left point in the destination.
	 * @param flip
	 *            the flip to apply.
	 * @param alpha
	 *            the opacity level to apply to the region.
	 */
	void drawFlippedImage(MicroUIGraphicsContext gc, MicroUIImage img, int regionX, int regionY, int width, int height, int x,
			int y, DrawingFlip flip, int alpha);

	/**
	 * Draws an image applying a free rotation (0 to 360 degrees).
	 * <p>
	 * The rotation is specified by the center and the angle. The reference point is the graphical object top-left
	 * corner. The rotation point is relative where the graphical object will be drawn.
	 * <p>
	 * This method uses the nearest neighbor algorithm to render the content. This algorithm is faster than bilinear
	 * algorithm but its rendering is faster.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param img
	 *            the MicroUI Image to draw.
	 * @param x
	 *            the x coordinate of the image reference anchor top-left point.
	 * @param y
	 *            the y coordinate of the image reference anchor top-left point.
	 * @param rotationX
	 *            the x coordinate of the rotation center.
	 * @param rotationY
	 *            the y coordinate of the rotation center.
	 * @param angle
	 *            the rotation angle.
	 * @param alpha
	 *            the opacity level to apply to the region.
	 */
	void drawRotatedImageNearestNeighbor(MicroUIGraphicsContext gc, MicroUIImage img, int x, int y, int rotationX,
			int rotationY, float angle, int alpha);

	/**
	 * Draws an image applying a free rotation (0 to 360 degrees).
	 * <p>
	 * The rotation is specified by the center and the angle. The reference point is the graphical object top-left
	 * corner. The rotation point is relative where the graphical object will be drawn.
	 * <p>
	 * This method uses the bilinear algorithm to render the content. This algorithm performs better rendering than
	 * nearest neighbor algorithm but it is slower to apply.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param img
	 *            the MicroUI Image to draw.
	 * @param x
	 *            the x coordinate of the image reference anchor top-left point.
	 * @param y
	 *            the y coordinate of the image reference anchor top-left point.
	 * @param rotationX
	 *            the x coordinate of the rotation center.
	 * @param rotationY
	 *            the y coordinate of the rotation center.
	 * @param angle
	 *            the rotation angle.
	 * @param alpha
	 *            the opacity level to apply to the region.
	 */
	void drawRotatedImageBilinear(MicroUIGraphicsContext gc, MicroUIImage img, int x, int y, int rotationX, int rotationY,
			float angle, int alpha);

	/**
	 * Draws an image applying a scaling.
	 * <p>
	 * This method uses the nearest neighbor algorithm to render the content. This algorithm is faster than bilinear
	 * algorithm but its rendering is faster.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param img
	 *            the MicroUI Image to draw.
	 * @param x
	 *            the x coordinate of the image reference anchor top-left point.
	 * @param y
	 *            the y coordinate of the image reference anchor top-left point.
	 * @param factorX
	 *            scaling X factor.
	 * @param factorY
	 *            scaling Y factor.
	 * @param alpha
	 *            the opacity level to apply to the region.
	 */
	void drawScaledImageNearestNeighbor(MicroUIGraphicsContext gc, MicroUIImage img, int x, int y, float factorX,
			float factorY, int alpha);

	/**
	 * Draws an image applying a scaling.
	 * <p>
	 * This method uses the bilinear algorithm to render the content. This algorithm performs better rendering than
	 * nearest neighbor algorithm but it is slower to apply.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param img
	 *            the MicroUI Image to draw.
	 * @param x
	 *            the x coordinate of the image reference anchor top-left point.
	 * @param y
	 *            the y coordinate of the image reference anchor top-left point.
	 * @param factorX
	 *            scaling X factor.
	 * @param factorY
	 *            scaling Y factor.
	 * @param alpha
	 *            the opacity level to apply to the region.
	 */
	void drawScaledImageBilinear(MicroUIGraphicsContext gc, MicroUIImage img, int x, int y, float factorX, float factorY,
			int alpha);

}
