/*
 * Java
 *
 * Copyright 2019 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.net.util.connectivity;

import android.net.ConnectivityManager;
import android.net.ConnectivityManager.NetworkCallback;
import android.net.Network;
import android.net.NetworkCapabilities;
import android.net.NetworkInfo;
import ej.annotation.NonNull;

/**
 * Utilitary classes to handle connectivity manager.
 */
public final class ConnectivityUtil {

	private ConnectivityUtil() {
		// Forbid instantiation.
	}

	/**
	 * Registers the callback as a default network callback, then call the callback depending on the current state.
	 *
	 * @param manager
	 *            the manager to use.
	 * @param callback
	 *            the callback to use.
	 * @see ConnectivityManager#registerDefaultNetworkCallback(NetworkCallback)
	 * @see #callCurrentState(ConnectivityManager, NetworkCallback)
	 */
	public static void registerAndCall(@NonNull ConnectivityManager manager, @NonNull NetworkCallback callback) {
		manager.registerDefaultNetworkCallback(callback);
		callCurrentState(manager, callback);
	}

	/**
	 * Checks the manager state and call the callbacks depending on it.
	 *
	 * @param manager
	 *            the manager to use.
	 * @param callback
	 *            the callback to use.
	 */
	public static void callCurrentState(@NonNull ConnectivityManager manager, @NonNull NetworkCallback callback) {
		Network activeNetwork = manager.getActiveNetwork();
		NetworkInfo networkInfo = manager.getNetworkInfo(activeNetwork);
		NetworkCapabilities capability = manager.getNetworkCapabilities(activeNetwork);
		if (networkInfo != null && networkInfo.isConnected()) {
			callback.onAvailable(activeNetwork);
		} else {
			callback.onLost(activeNetwork);
		}

		if (capability != null) {
			callback.onCapabilitiesChanged(activeNetwork, capability);
		}
	}

}
