/*
 * Java
 *
 * Copyright 2025 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.kf.util.control.fs;

import com.microej.kf.util.module.SandboxedModule;
import com.microej.kf.util.storage.SandboxedStorage;
import ej.basictool.ArrayTools;
import ej.kf.Kernel;
import ej.kf.Module;
import ej.service.ServiceFactory;
import ej.storage.Storage;

import java.io.File;
import java.util.logging.Logger;

/**
 * The file system resources controller.
 *
 * <p>
 * It controls the numbers of opened files, the storage size and the file system bandwidth. By default, there are no
 * limits on the file system usage.
 *
 * <p>
 * When an application tries to perform a file system operation (open/read/write), the controller checks if a file
 * system limit setting is defined for the application and denies the operation if its execution will lead to exceeding
 * one of the limits.
 * <p>
 * It also notifies the registered listeners when an operation is denied or when a limit is reached.
 */
public class FileSystemResourcesController {

	/** The maximum number of opened files allowed. */
	private int maxOpenedFiles;

	/** The maximum storage size allowed. */
	private long maxStorageSize;

	 /** The current storage size used. */
	private long currentStorageSize;

	 /** The current opened files. */
	private File[] currentOpenedFiles;

	private final Logger logger;

	private final Module module;

	/**
	 * Creates an instance of the File System controller.
	 *
	 * @param module
	 * 		The module which performs the file system controls.
	 */
	public FileSystemResourcesController(Module module) {
		this.logger = Logger.getLogger(FileSystemResourcesController.class.getName());
		this.module = module;

		reset();
	}

	/**
	 * Resets the controller.
	 */
	public synchronized void reset() {
		SandboxedStorage storage = (SandboxedStorage) ServiceFactory.getRequiredService(Storage.class);
		this.currentStorageSize = computeStorageSize(new File(storage.getRootFolder(this.module)));

		this.maxOpenedFiles = -1;
		this.maxStorageSize = -1;
		this.currentOpenedFiles = new File[0];
	}

	/////////////////////////////////////////
	// Open file action
	/////////////////////////////////////////

	/**
	 * Called when the execution of the open file action is about to start.
	 * <p>
	 * It checks if the action is allowed to be executed.
	 *
	 * @param module
	 * 		the module which performs the action.
	 * @param openFile
	 * 		the open file action.
	 * @throws SecurityException
	 * 		if the action is not allowed to be performed.
	 */
	public synchronized void onStart(SandboxedModule module, OpenFile openFile) throws SecurityException {

		if (this.maxOpenedFiles < 0 || this.currentOpenedFiles.length < this.maxOpenedFiles) {
			return;
		}
		this.logger.fine("Opened files limit reached (only " + this.maxOpenedFiles + " opened files allowed)"); // $NON-NLS-1$
		throw new OpenedFilesLimitException();
	}

	/**
	 * Called when the execution of the given open file action is finished.
	 *
	 * @param module
	 * 		the module which performs the given open file action.
	 * @param openFile
	 * 		the open file action.
	 * @param withSuccess
	 * 		true if the execution of the action was ended with success; false otherwise.
	 */
	public synchronized void onEnd(SandboxedModule module, OpenFile openFile, boolean withSuccess) {
		if (withSuccess) {
			Kernel.enter();
			this.currentOpenedFiles = ArrayTools.add(this.currentOpenedFiles, openFile.getFile());
			if (openFile.isWriteMode() && openFile.isOverwritten()) {
				// In the case where the file is opened in write mode and overwritten,
				// it's like the file has been deleted and then created with empty content.
				// We need to update the current storage size by subtracting the overwritten size.

				this.currentStorageSize -= openFile.getOverwritingSize();
				this.currentStorageSize = (this.currentStorageSize < 0) ? 0 : this.currentStorageSize;
			}
		}
	}

	/////////////////////////////////////////
	// Close file action
	/////////////////////////////////////////

	/**
	 * Called when the execution of the close file action is about to start.
	 * <p>
	 * It checks if the action is allowed to be executed.
	 *
	 * @param module
	 * 		the module which performs the action.
	 * @param closeFile
	 * 		the close file action.
	 * @throws SecurityException
	 * 		if the action is not allowed to be performed.
	 */
	public synchronized void onStart(SandboxedModule module, CloseFile closeFile) throws SecurityException {
		// Called by closeFile.onStart().
	}

	/**
	 * Called when the execution of the given close file action is finished.
	 *
	 * @param module
	 * 		the application which performs the given close file action.
	 * @param closeFile
	 * 		the close file action.
	 * @param withSuccess
	 * 		true if the execution of the action was ended with success; false otherwise.
	 */
	public synchronized void onEnd(SandboxedModule module, CloseFile closeFile, boolean withSuccess) {
		if (withSuccess) {
			Kernel.enter();
			this.currentOpenedFiles = ArrayTools.remove(this.currentOpenedFiles, closeFile.getFile());
		}
	}

	/////////////////////////////////////////
	// Write file action
	/////////////////////////////////////////

	/**
	 * Called when the execution of the write file action is about to start.
	 * <p>
	 * It checks if the action is allowed to be executed and throttles (delays) it if needed.
	 * <p>
	 *
	 * @param module
	 * 		the module which performs the write file action.
	 * @param writeFile
	 * 		the write file action.
	 * @throws SecurityException
	 * 		if the action is not allowed to be performed.
	 */
	public synchronized void onStart(SandboxedModule module, WriteFile writeFile) throws SecurityException {
		int nbBytes = writeFile.getNbBytes();

		Kernel.enter();

		if ((this.maxStorageSize < 0) || ((this.currentStorageSize + nbBytes) <= this.maxStorageSize)) {
			return;
		}
		this.logger.fine(
				"Cannot write file: Max storage size will be exceeded. New forecasted storage size: " // NOSONAR
						+ (this.currentStorageSize + nbBytes) + "; Max storage size allowed: " // NOSONAR
						+ this.maxStorageSize + ")");
		throw new StorageLimitException();
	}

	/**
	 * Called when the execution of the given write file action is finished.
	 *
	 * @param module
	 * 		the module which performs the given write file action.
	 * @param writeFile
	 * 		the write file action.
	 * @param withSuccess
	 * 		true if the execution of the action was ended with success; false otherwise.
	 */
	public synchronized void onEnd(SandboxedModule module, WriteFile writeFile, boolean withSuccess) {
		int writeBytes = writeFile.getNbBytes();

		if (withSuccess) {
			this.currentStorageSize += writeBytes;
		}
	}

	/////////////////////////////////////////
	// SetLength file action
	/////////////////////////////////////////

	/**
	 * Called when the execution of the write file action is about to start.
	 * <p>
	 * It checks if the action is allowed to be executed and throttles (delays) it if needed.
	 * <p>
	 *
	 * @param module
	 * 		the module which performs the write file action.
	 * @param setLengthFile
	 * 		the write file action.
	 * @throws SecurityException
	 * 		if the action is not allowed to be performed.
	 */
	public synchronized void onStart(SandboxedModule module, SetLengthRandomAccessFile setLengthFile)
			throws SecurityException {
		long nbBytes = setLengthFile.getNbBytes();

		Kernel.enter();

		if ((this.maxStorageSize < 0) || ((this.currentStorageSize + nbBytes) <= this.maxStorageSize)) {
			return;
		}

		this.logger.fine(
				"Cannot write file: Max storage size will be exceeded. New forecasted storage size: " // NOSONAR
						+ (this.currentStorageSize + nbBytes) + "; Max storage size allowed: " // NOSONAR
						+ this.maxStorageSize + ")");
		throw new StorageLimitException();
	}

	/**
	 * Called when the execution of the given write file action is finished.
	 *
	 * @param module
	 * 		the module which performs the given write file action.
	 * @param setLengthFile
	 * 		the write file action.
	 * @param withSuccess
	 * 		true if the execution of the action was ended with success; false otherwise.
	 */
	public synchronized void onEnd(SandboxedModule module, SetLengthRandomAccessFile setLengthFile,
			boolean withSuccess) {
		long writeBytes = setLengthFile.getNbBytes();

		if (withSuccess) {
			this.currentStorageSize += writeBytes;
		}
	}

	/////////////////////////////////////////
	// Write file action
	/////////////////////////////////////////

	/**
	 * Called when the execution of the write file action is about to start.
	 * <p>
	 * It checks if the action is allowed to be executed and throttles (delays) it if needed.
	 * <p>
	 *
	 * @param module
	 * 		the module which performs the write file action.
	 * @param writeFile
	 * 		the write file action.
	 * @throws SecurityException
	 * 		if the action is not allowed to be performed.
	 */
	public synchronized void onStart(SandboxedModule module, WriteRandomAccessFile writeFile) throws SecurityException {
		long nbBytes = writeFile.getNbBytes();

		Kernel.enter();

		if ((this.maxStorageSize < 0) || ((this.currentStorageSize + nbBytes) <= this.maxStorageSize)) {
			return;
		}
		this.logger.fine(
				"Cannot write file: Max storage size will be exceeded. New forecasted storage size: " // NOSONAR
						+ (this.currentStorageSize + nbBytes) + "; Max storage size allowed: " // NOSONAR
						+ this.maxStorageSize + ")");

		throw new StorageLimitException();
	}

	/**
	 * Called when the execution of the given write file action is finished.
	 *
	 * @param module
	 * 		the module which performs the given write file action.
	 * @param writeFile
	 * 		the write file action.
	 * @param withSuccess
	 * 		true if the execution of the action was ended with success; false otherwise.
	 */
	public synchronized void onEnd(SandboxedModule module, WriteRandomAccessFile writeFile, boolean withSuccess) {
		long nbBytes = writeFile.getNbBytes();

		if (withSuccess) {
			this.currentStorageSize += nbBytes;
		}
	}

	/////////////////////////////////////////
	// Delete file action
	/////////////////////////////////////////

	/**
	 * Called when the execution of the delete file action is about to start.
	 * <p>
	 * It checks if the action is allowed to be executed.
	 * <p>
	 *
	 * @param module
	 * 		the module which performs the delete file action
	 * @param deleteFile
	 * 		the delete file action
	 * @throws SecurityException
	 * 		if the action is not allowed to be performed.
	 */
	public synchronized void onStart(SandboxedModule module, DeleteFile deleteFile) throws SecurityException {
		// Called by deleteFile.onStart().
	}

	/**
	 * Called when the execution of the given delete file action is finished.
	 *
	 * @param module
	 * 		the module which performs the given delete file action.
	 * @param deleteFile
	 * 		the delete file action.
	 * @param withSuccess
	 * 		true if the execution of the action was ended with success; false otherwise.
	 */
	public synchronized void onEnd(SandboxedModule module, DeleteFile deleteFile, boolean withSuccess) {
		if (withSuccess) {
			this.currentStorageSize -= deleteFile.getNbBytes();
			this.currentStorageSize = (this.currentStorageSize < 0) ? 0 : this.currentStorageSize;
		}
	}

	/**
	 * Sets the maximum file system storage size (in bytes) allowed.
	 *
	 * @param maxStorageSize
	 * 		the maximum file system storage size. A negative value means there is no limit.
	 */
	public synchronized void setMaxStorageSize(long maxStorageSize) {
		this.maxStorageSize = maxStorageSize;
	}

	/**
	 * Sets the maximum number of files allowed to be opened.
	 *
	 * @param maxOpenFiles
	 * 		the maximum number of files allowed to be opened. A negative value means there is no limit.
	 */
	public synchronized void setMaxOpenedFiles(int maxOpenFiles) {
		this.maxOpenedFiles = maxOpenFiles;
	}

	/**
	 * Returns the current storage size.
	 *
	 * @return the current storage size.
	 */
	public synchronized long getStorageSize() {
		return this.currentStorageSize;
	}

	/**
	 * Returns the current opened files.
	 *
	 * @return the current opened files.
	 */
	public synchronized File[] getOpenedFiles() {
		return this.currentOpenedFiles.clone();
	}

	/**
	 * Computes the storage size of a given module.
	 */
	private long computeStorageSize(File dir) {
		long length = 0;

		File[] files = dir.listFiles();
		if (files != null) {
			for (File file : files) {
				if (file.isFile())
					length += file.length();
				else
					length += computeStorageSize(file);
			}
		}
		return length;
	}

	/**
	 * @return the maximum storage size in bytes. A negative value means there is no limit.
	 */
	public synchronized long getMaxStorageSize() {
		return this.maxStorageSize;
	}

	/**
	 * @return the maximum number of files allowed to be opened. A negative value means there is no limit.
	 */
	public synchronized int getMaxOpenedFiles() {
		return this.maxOpenedFiles;
	}
}
