/*
 * Copyright (c) 1994, 2013, Oracle and/or its affiliates. All rights reserved.
 * Copyright (C) 2014-2020 MicroEJ Corp. - EDC compliance and optimizations.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package sun.net.www.http;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintStream;
import java.io.UnsupportedEncodingException;
import java.net.Socket;
import java.net.SocketException;
import java.net.SocketPermission;
import java.net.SocketTimeoutException;
import java.net.URL;
import java.util.ArrayList;

import com.microej.http.util.HttpBufferedStreamFactory;

import ej.annotation.Nullable;
import sun.net.NetworkClient;
import sun.net.www.MessageHeader;
import sun.net.www.protocol.http.HttpURLConnection;

/**
 * @author Herb Jellinek
 * @author Dave Brown
 */
public class HttpClient extends NetworkClient {

	// Http requests we send
	@Nullable
	MessageHeader requests;

	// Http data we send with the headers
	@Nullable
	PosterOutputStream poster = null;

	// true if we are in streaming mode (fixed length or chunked)
	boolean streaming;

	// if we've had one io error
	boolean failedOnce = false;

	/** Response code for CONTINUE */
	private boolean ignoreContinue = true;
	private static final int HTTP_CONTINUE = 100;

	/** Default port number for http daemons. REMIND: make these private */
	static final int httpPortNumber = 80;

	/** return default port number (subclasses may override) */
	protected int getDefaultPort() {
		return httpPortNumber;
	}

	// target host, port for the URL
	@Nullable
	protected String host;
	protected int port;

	// retryPostProp is true by default so as to preserve behavior
	// from previous releases.
	private static boolean retryPostProp = true;

	/** Url being fetched. */
	protected URL url;

	/* if set, the client will be reused and must not be put in cache */
	public boolean reuse = false;

	/**
	 * A NOP method kept for backwards binary compatibility
	 *
	 * @deprecated -- system properties are no longer cached.
	 */
	@Deprecated
	public static synchronized void resetProperties() {
	}

	static {
		String retryPost = System.getProperty("sun.net.http.retryPost");
		if (retryPost != null) {
			retryPostProp = Boolean.valueOf(retryPost).booleanValue();
		} else {
			retryPostProp = true;
		}

	}

	protected HttpClient() {
	}

	protected HttpClient(URL url, int to) throws IOException {
		this.host = url.getHost();
		this.url = url;
		this.port = url.getPort();
		if (this.port == -1) {
			this.port = getDefaultPort();
		}
		setConnectTimeout(to);

		openServer();
	}

	/*
	 * This class has no public constructor for HTTP. This method is used to get an HttpClient to the specifed URL. If
	 * there's currently an active HttpClient to that server/port, you'll get that one.
	 */
	public static HttpClient New(URL url) throws IOException {
		return HttpClient.New(url, -1, null);
	}

	public static HttpClient New(URL url, int to, @Nullable HttpURLConnection httpuc) throws IOException {
		HttpClient ret = new HttpClient(url, to);
		return ret;
	}

	/*
	 * return it to the cache as still usable, if: 1) It's keeping alive, AND 2) It still has some connections left, AND
	 * 3) It hasn't had a error (PrintStream.checkError()) 4) It hasn't timed out
	 *
	 * If this client is not keepingAlive, it should have been removed from the cache in the parseHeaders() method.
	 */

	public void finished() {
		if (this.reuse) {
			return;
		}
		this.poster = null;
		closeServer();
	}

	protected synchronized boolean available() {
		boolean available = true;
		int old = -1;

		try {
			try {
				old = this.serverSocket.getSoTimeout();
				this.serverSocket.setSoTimeout(1);
				InputStream tmpbuf = HttpBufferedStreamFactory.newInputStream(this.serverSocket.getInputStream());
				int r = tmpbuf.read();
				if (r == -1) {
					available = false;
				}
			} catch (SocketTimeoutException e) {
			} finally {
				if (old != -1) {
					this.serverSocket.setSoTimeout(old);
				}
			}
		} catch (IOException e) {
			available = false;
		}
		return available;
	}

	/*
	 * Close an idle connection to this URL (if it exists in the cache).
	 */
	public void closeIdleConnection() {
	}

	/*
	 * We're very particular here about what our InputStream to the server looks like for reasons that are apparent if
	 * you can decipher the method parseHTTP(). That's why this method is overidden from the superclass.
	 */
	@Override
	public void openServer(String server, int port) throws IOException {
		Socket serverSocket = this.serverSocket = doConnect(server, port);
		try {
			OutputStream out = serverSocket.getOutputStream();
			this.serverOutput = new PrintStream(HttpBufferedStreamFactory.newOutputStream(out), false, encoding);
		} catch (UnsupportedEncodingException e) {
			throw new InternalError(encoding + " encoding not found");
		}
	}

	/*
	 */
	protected synchronized void openServer() throws IOException {

		SecurityManager security = System.getSecurityManager();

		if (security != null) {
			checkConnect(security, this.host, this.port);
		}

		if (this.url.getProtocol().equals("http") || this.url.getProtocol().equals("https")) {

			// make direct connection
			openServer(this.host, this.port);
			return;

		} else {
			/*
			 * we're opening some other kind of url, most likely an ftp url.
			 */
			// make direct connection
			super.openServer(this.host, this.port);
			return;
		}
	}

	public String getURLFile() throws IOException {

		String fileName = this.url.getFile();
		if ((fileName == null) || (fileName.length() == 0)) {
			fileName = "/";
		}

		if (fileName.indexOf('\n') == -1) {
			return fileName;
		} else {
			throw new java.net.MalformedURLException("Illegal character in URL");
		}
	}

	/**
	 * @deprecated
	 */
	@Deprecated
	public void writeRequests(MessageHeader head) {
		MessageHeader requests = this.requests = head;
		requests.print(this.serverOutput);
		this.serverOutput.flush();
	}

	public void writeRequests(MessageHeader head, PosterOutputStream pos) throws IOException {
		MessageHeader requests = this.requests = head;
		requests.print(this.serverOutput);
		PosterOutputStream poster = this.poster = pos;
		if (poster != null) {
			poster.writeTo(this.serverOutput);
		}
		this.serverOutput.flush();
	}

	public void writeRequests(MessageHeader head, PosterOutputStream pos, boolean streaming) throws IOException {
		this.streaming = streaming;
		writeRequests(head, pos);
	}

	/**
	 * Parse the first line of the HTTP request. It usually looks something like: "HTTP/1.0 &lt;number&gt; comment\r\n".
	 **/
	public boolean parseHTTP(MessageHeader responses, HttpURLConnection httpuc) throws IOException {
		/*
		 * If "HTTP/*" is found in the beginning, return true. Let HttpURLConnection parse the mime header itself.
		 *
		 * If this isn't valid HTTP, then we don't try to parse a header out of the beginning of the response into the
		 * responses, and instead just queue up the output stream to it's very beginning. This seems most reasonable,
		 * and is what the NN browser does.
		 */

		try {
			this.serverInput = HttpBufferedStreamFactory.newInputStream(this.serverSocket.getInputStream());
			return (parseHTTPHeader(responses, httpuc));
		} catch (SocketTimeoutException stex) {
			// We don't want to retry the request when the app. sets a timeout
			// but don't close the server if timeout while waiting for 100-continue
			if (this.ignoreContinue) {
				closeServer();
			}
			throw stex;
		} catch (IOException e) {
			closeServer();
			MessageHeader requests = this.requests;
			if (!this.failedOnce && requests != null) {
				this.failedOnce = true;
				if (getRequestMethod().equals("CONNECT")
						|| (httpuc.getRequestMethod().equals("POST") && (!retryPostProp || this.streaming))) {
					// do not retry the request
				} else {
					// try once more
					openServer();
					writeRequests(requests, this.poster);
					return parseHTTP(responses, httpuc);
				}
			}
			throw e;
		}

	}

	private boolean parseHTTPHeader(MessageHeader responses, HttpURLConnection httpuc) throws IOException {
		/*
		 * If "HTTP/*" is found in the beginning, return true. Let HttpURLConnection parse the mime header itself.
		 *
		 * If this isn't valid HTTP, then we don't try to parse a header out of the beginning of the response into the
		 * responses, and instead just queue up the output stream to it's very beginning. This seems most reasonable,
		 * and is what the NN browser does.
		 */

		boolean ret = false;
		byte[] b = new byte[8];

		try {
			int nread = 0;
			this.serverInput.mark(10);
			while (nread < 8) {
				int r = this.serverInput.read(b, nread, 8 - nread);
				if (r < 0) {
					break;
				}
				nread += r;
			}
			ret = b[0] == 'H' && b[1] == 'T' && b[2] == 'T' && b[3] == 'P' && b[4] == '/' && b[5] == '1' && b[6] == '.';
			this.serverInput.reset();
			if (ret) { // is valid HTTP - response started w/ "HTTP/1."
				responses.parseHeader(this.serverInput);

				// we've finished parsing http headers
			} else if (nread != 8) {
				MessageHeader requests = this.requests;
				if (!this.failedOnce && requests != null) {
					this.failedOnce = true;
					if (getRequestMethod().equals("CONNECT")
							|| (httpuc.getRequestMethod().equals("POST") && (!retryPostProp || this.streaming))) {
						// do not retry the request
					} else {
						closeServer();
						openServer();
						writeRequests(requests, this.poster);
						return parseHTTP(responses, httpuc);
					}
				}
				throw new SocketException("Unexpected end of file from server");
			} else {
				// we can't vouche for what this is....
				responses.set("Content-type", "unknown/unknown");
			}
		} catch (IOException e) {
			throw e;
		}

		int code = -1;
		try {
			String resp;
			resp = responses.getValue(0);
			/*
			 * should have no leading/trailing LWS expedite the typical case by assuming it has form
			 * "HTTP/1.x <WS> 2XX <mumble>"
			 */
			int ind;
			ind = resp.indexOf(' ');
			while (resp.charAt(ind) == ' ') {
				ind++;
			}
			code = Integer.parseInt(resp.substring(ind, ind + 3));
		} catch (Exception e) {
		}

		if (code == HTTP_CONTINUE && this.ignoreContinue) {
			responses.reset();
			return parseHTTPHeader(responses, httpuc);
		}

		/*
		 * Set things up to parse the entity body of the reply. We should be smarter about avoid pointless work when the
		 * HTTP method and response code indicate there will be no entity body to parse.
		 */
		String te = responses.findValue("Transfer-Encoding");
		if (te != null && te.equalsIgnoreCase("chunked")) {
			this.serverInput = new ChunkedInputStream(this.serverInput, this, responses);
			this.failedOnce = false;
		}

		return ret;
	}

	@Nullable
	public synchronized InputStream getInputStream() {
		return this.serverInput;
	}

	@Nullable
	public OutputStream getOutputStream() {
		return this.serverOutput;
	}

	@Override
	public String toString() {
		return getClass().getName() + "(" + this.url + ")";
	}

	String getRequestMethod() {
		MessageHeader requests = this.requests;
		if (requests != null) {
			String requestLine = requests.getKey(0);
			if (requestLine != null) {
				return splitWhitespace(requestLine)[0];
			}
		}
		return "";
	}

	/**
	 * String.split("\\s+") &lt;=&gt; split()
	 */
	public static String[] splitWhitespace(String str) {
		ArrayList<String> result = new ArrayList<>();
		int length = str.length();
		StringBuilder current = new StringBuilder();
		for (int i = -1; ++i < length;) {
			char c = str.charAt(i);
			if (c == '\t' || c == '\n' || c == '\u000B' || c == '\f' || c == '\r') {
				// whitespace
				if (current.length() > 0) {
					// flush
					result.add(current.toString());
					current = new StringBuilder();
				}
			} else {
				current.append(c);
			}
		}

		if (current.length() > 0) {
			// flush
			result.add(current.toString());
			current = new StringBuilder();
		}

		return result.toArray(new String[result.size()]);
	}

	public void setDoNotRetry(boolean value) {
		// failedOnce is used to determine if a request should be retried.
		this.failedOnce = value;
	}

	public void setIgnoreContinue(boolean value) {
		this.ignoreContinue = value;
	}

	/* Use only on connections in error. */
	@Override
	public void closeServer() {
		try {
			this.serverSocket.close();
		} catch (Exception e) {
		}
	}

	public static final String SOCKET_RESOLVE_ACTION = "resolve";
	public static final String SOCKET_CONNECT_ACTION = "connect";

	public static void checkConnect(SecurityManager sm, String host, int port) {
		if (host == null) {
			throw new NullPointerException("host can't be null");
		}
		if (!host.startsWith("[") && host.indexOf(':') != -1) {
			host = "[" + host + "]";
		}
		if (port == -1) {
			sm.checkPermission(new SocketPermission(host, SOCKET_RESOLVE_ACTION));
		} else {
			sm.checkPermission(new SocketPermission(host + ":" + port, SOCKET_CONNECT_ACTION));
		}
	}
}
