/*
 * Java
 *
 * Copyright 2022-2023 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.microvg.test;

import static org.junit.Assert.assertEquals;

import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

import ej.microui.display.Colors;
import ej.microui.display.Display;
import ej.microui.display.GraphicsContext;
import ej.microui.display.Painter;
import ej.microvg.BufferedVectorImage;
import ej.microvg.Matrix;
import ej.microvg.Path;
import ej.microvg.VectorGraphicsPainter;

/**
 * Tests the drawing of MicroVG shapes into a BufferedVectorImage and using clip(s).
 */
public class TestBufferedVectorImageClip {

	/**
	 * Starts MicroUI.
	 */
	@BeforeClass
	public static void pre() {
		TestUtilities.startMicroUI();
	}

	/**
	 * Stops MicroUI.
	 */
	@AfterClass
	public static void post() {
		TestUtilities.stopMicroUI();
	}

	/**
	 * Resets the content of the screen to black.
	 */
	@Before
	public static void preTest() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		g.reset();
		g.setColor(Colors.BLACK);
		Painter.fillRectangle(g, 0, 0, display.getWidth(), display.getHeight());
	}

	/**
	 * Tests no clip
	 */
	@Test
	public static void testNoClip() {

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		BufferedVectorImage image = new BufferedVectorImage(20, 20);
		GraphicsContext ig = image.getGraphicsContext();

		g.translate(200, 200);

		ig.setColor(Colors.RED);
		drawPath(ig);

		VectorGraphicsPainter.drawImage(g, image, new Matrix());
		display.flush();

		// source: 20 x 20
		// dest: 20 x 20 at (200,200)
		checkAreas(0, 0, 20, 20);
	}

	/**
	 * Tests clip on destination
	 */
	@Test
	public static void testClipOnDestination() {

		int x = 1;
		int y = 1;
		int w = 10;
		int h = 10;

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		BufferedVectorImage image = new BufferedVectorImage(2 * w, 2 * h);
		GraphicsContext ig = image.getGraphicsContext();

		g.translate(200, 200);
		g.setClip(x, y, w, h);

		ig.setColor(Colors.RED);
		drawPath(ig);

		VectorGraphicsPainter.drawImage(g, image, new Matrix());
		display.flush();

		// source: 20 x 20
		// dest: 10 x 10 at (201,201)
		checkAreas(x, y, w, h);
	}

	/**
	 * Tests clip on source
	 */
	@Test
	public static void testClipOnSource() {

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		BufferedVectorImage image = new BufferedVectorImage(20, 20);
		GraphicsContext ig = image.getGraphicsContext();

		g.translate(200, 200);

		ig.setColor(Colors.RED);
		ig.setClip(2, 2, 5, 5);
		drawPath(ig);

		VectorGraphicsPainter.drawImage(g, image, new Matrix());
		display.flush();

		// source: 5 x 5
		// dest: 5 x 5 at (202,202)
		checkAreas(2, 2, 5, 5);
	}

	/**
	 * Tests clip on source and destination
	 */
	@Test
	public static void testClipOnSourceAndDestination() {

		int x = 1;
		int y = 10;
		int w = 25;
		int h = 5;

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		BufferedVectorImage image = new BufferedVectorImage(60, 60);
		GraphicsContext ig = image.getGraphicsContext();

		g.translate(200, 200);
		g.setClip(x, y, w, h);

		ig.setColor(Colors.RED);
		ig.setClip(5, 5, 10, 10);
		drawPath(ig);

		VectorGraphicsPainter.drawImage(g, image, new Matrix());
		display.flush();

		// source: 10 x 10
		// dest: 10 x 5 at (205,210)
		checkAreas(5, y, 10, 5);
	}

	/**
	 * Tests scaling on source
	 */
	@Test
	public static void testScalingOnSource() {

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		BufferedVectorImage image = new BufferedVectorImage(20, 20);
		GraphicsContext ig = image.getGraphicsContext();

		g.translate(200, 200);

		ig.setColor(Colors.RED);
		drawPath(ig);

		Matrix matrix = new Matrix();
		matrix.setScale(0.25f, 0.25f);
		VectorGraphicsPainter.drawImage(g, image, matrix);
		display.flush();

		// source: 20 x 20
		// dest: 5 x 5 at (200,200)
		checkAreas(0, 0, 5, 5);
	}

	/**
	 * Tests scaling on source and clip on destination
	 */
	@Test
	public static void testScalingOnSourceAndClipOnDestination() {

		int x = 1;
		int y = 1;
		int w = 10;
		int h = 10;

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		BufferedVectorImage image = new BufferedVectorImage(2 * w, 2 * h);
		GraphicsContext ig = image.getGraphicsContext();

		g.translate(200, 200);
		g.setClip(x, y, w, h);

		ig.setColor(Colors.RED);
		drawPath(ig);

		Matrix matrix = new Matrix();
		matrix.setScale(0.25f, 0.25f);
		VectorGraphicsPainter.drawImage(g, image, matrix);
		display.flush();

		// source: 20 x 20
		// dest: 4 x 4 at (201,201)
		checkAreas(x, y, 4, 4);
	}

	/**
	 * Tests scaling and clip on source
	 */
	@Test
	public static void testScalingAndClipOnSource() {

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		BufferedVectorImage image = new BufferedVectorImage(60, 60);
		GraphicsContext ig = image.getGraphicsContext();

		g.translate(200, 200);

		ig.setColor(Colors.RED);
		ig.setClip(12, 12, 32, 32);
		drawPath(ig);

		Matrix matrix = new Matrix();
		matrix.setScale(0.25f, 0.25f);
		VectorGraphicsPainter.drawImage(g, image, matrix);
		display.flush();

		// source: 60 x 60
		// dest: 5 x 5 at (203,203) // 3 = 12 / 4
		checkAreas(3, 3, 8, 8);
	}

	/**
	 * Tests scaling on source and clip on source and destination
	 */
	@Test
	public static void testScalingOnSourceAndClipOnSourceAndDestination() {

		int x = 1;
		int y = 1;
		int w = 5;
		int h = 5;

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		BufferedVectorImage image = new BufferedVectorImage(60, 60);
		GraphicsContext ig = image.getGraphicsContext();

		g.translate(200, 200);
		g.setClip(x, y, w, h);

		ig.setColor(Colors.RED);
		ig.setClip(12, 12, 32, 32);
		drawPath(ig);

		Matrix matrix = new Matrix();
		matrix.setScale(0.25f, 0.25f);
		VectorGraphicsPainter.drawImage(g, image, matrix);
		display.flush();

		// source: 60 x 60
		// dest: 3 x 3 at (203,203) // 3 = 12 / 4
		checkAreas(3, 3, 3, 3);
	}

	private static void checkAreas(int x, int y, int w, int h) {
		int errors = 0;
		errors += TestUtilities.checkPeripheralArea("clip over inside", Colors.RED, x + 1, y + 1, w - 2, h - 2, 1, 0, //$NON-NLS-1$
				false);
		errors += TestUtilities.checkPeripheralArea("clip over outside", Colors.BLACK, x, y, w, h, 1, 0, false); //$NON-NLS-1$
		assertEquals(0, errors);
	}

	private static void drawPath(GraphicsContext g) {
		Path path = new Path();
		int w = g.getWidth();
		int h = g.getHeight();
		path.lineTo(w, 0);
		path.lineTo(w, h);
		path.lineTo(0, h);
		path.close();
		VectorGraphicsPainter.fillPath(g, path, 0, 0);
	}
}
