/*
 * Java
 *
 * Copyright 2009-2021 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.hoka.tcp;

import static ej.hoka.http.HttpConstants.TAB;

import java.io.IOException;
import java.net.InetAddress;
import java.net.ServerSocket;
import java.net.Socket;

import javax.net.ServerSocketFactory;

import ej.annotation.Nullable;
import ej.hoka.log.HokaLogger;

/**
 * TCP/IP server that stores incoming connections.
 * <p>
 * After starting this server with {@link #start()}, connections are available through
 * {@link #getNextStreamConnection()}.
 */
public class TcpServer {

	/**
	 * By default, server is configured to keep connection open during one minute if possible.
	 */
	private static final int DEFAULT_TIMEOUT_DURATION = 60000; // 60s

	/**
	 * Default host address returned when no address is found.
	 */
	private static final String DEFAULT_ADDRESS = "0.0.0.0"; //$NON-NLS-1$

	/**
	 * The port used by this server.
	 */
	private final int port;

	/**
	 * Maximum number of opened waiting connections.
	 */
	private final int maxOpenedConnections;

	/**
	 * The server socket factory used to start this server.
	 */
	private final ServerSocketFactory serverSocketFactory;

	/**
	 * The request timeout duration in milliseconds, 0 means no timeout (infinite persistent connections).
	 */
	private final int timeout;

	/**
	 * The server socket used by this server.
	 */
	@Nullable
	private ServerSocket serverSocket;

	/**
	 * The thread used by this server.
	 */
	@Nullable
	private Thread thread;

	/**
	 * Non growable circular queue of opened connections.
	 */
	private Socket[] streamConnections;

	/**
	 * Pointer to the last added item.
	 */
	private int lastAddedPtr;

	/**
	 * Pointer for the last read item.
	 */
	private int lastReadPtr;

	/**
	 * Constructs a new instance of {@link TcpServer} using environment's default socket factory.
	 *
	 * @param port
	 *            the port to use.
	 * @param maxOpenedConnections
	 *            the maximal number of simultaneously opened connections.
	 */
	public TcpServer(int port, int maxOpenedConnections) {
		this(port, maxOpenedConnections, ServerSocketFactory.getDefault());
	}

	/**
	 * Constructs a new instance of {@link TcpServer} using the {@link ServerSocketFactory}
	 * <code>serverSocketFactory</code>.
	 *
	 * @param port
	 *            the port to use.
	 * @param maxOpenedConnections
	 *            the maximal number of simultaneously opened connections.
	 * @param serverSocketFactory
	 *            the {@link ServerSocketFactory}.
	 */
	public TcpServer(int port, int maxOpenedConnections, ServerSocketFactory serverSocketFactory) {
		this(port, maxOpenedConnections, serverSocketFactory, DEFAULT_TIMEOUT_DURATION);
	}

	/**
	 * Constructs a new instance of {@link TcpServer} using the {@link ServerSocketFactory}
	 * <code>serverSocketFactory</code>.
	 *
	 * @param port
	 *            the port to use.
	 * @param maxOpenedConnections
	 *            the maximal number of simultaneously opened connections.
	 * @param serverSocketFactory
	 *            the {@link ServerSocketFactory}.
	 * @param timeout
	 *            the timeout of opened connections.
	 * @see Socket#setSoTimeout(int)
	 */
	public TcpServer(int port, int maxOpenedConnections, ServerSocketFactory serverSocketFactory, int timeout) {
		if (maxOpenedConnections <= 0 || timeout < 0) {
			throw new IllegalArgumentException();
		}

		this.port = port;
		this.maxOpenedConnections = maxOpenedConnections;
		this.serverSocketFactory = serverSocketFactory;
		this.timeout = timeout;
		this.streamConnections = new Socket[this.maxOpenedConnections + 1]; // always an empty index in order to
																			// distinguish between empty or full queue
	}

	/**
	 * Starts the {@link TcpServer}. The {@link TcpServer} can be started only once. Calling this method while the
	 * {@link TcpServer} is already running causes a {@link IllegalStateException}.
	 *
	 * @throws IOException
	 *             if an error occurs during the creation of the socket.
	 */
	public void start() throws IOException {
		if (!isStopped()) {
			throw new IllegalStateException();
		}

		this.streamConnections = new Socket[this.maxOpenedConnections + 1]; // always an empty index in order to
																			// distinguish between empty or full queue

		this.lastAddedPtr = 0;
		this.lastReadPtr = 0;

		this.serverSocket = this.serverSocketFactory.createServerSocket(this.port);

		this.thread = new Thread(newProcess(), getName());
		this.thread.start();

		String host = DEFAULT_ADDRESS;
		final InetAddress address = getInetAddress();
		if (address != null) {
			host = address.getHostAddress();
		}
		HokaLogger.instance.info("Server started " + host + ":" + getPort()); //$NON-NLS-1$ //$NON-NLS-2$
	}

	/**
	 * Stops the {@link TcpServer} and closes the connection.
	 */
	public void stop() {
		try {
			final ServerSocket server = this.serverSocket;
			this.serverSocket = null; // indicates the connection is being closed
			try {
				if (server != null) {
					server.close();
				}
			} catch (final IOException e) {
				HokaLogger.instance.error(e);
				return; // already closed
			}
			// join the end of the thread.
			final Thread t = this.thread;
			try {
				if (t != null) {
					t.join();
				}
			} catch (final InterruptedException e) {
				// nothing to do on interrupted exception
				HokaLogger.instance.error(e);
				t.interrupt(); // Restore interrupted state

			}
		} finally {
			this.thread = null; // to allow server to restart after being stopped
		}

		// awake all waiting threads
		synchronized (this.streamConnections) {
			this.streamConnections.notifyAll();
		}

		HokaLogger.instance.debug("server stopped"); //$NON-NLS-1$
	}

	/**
	 * Add a connection to the list of opened connections.
	 *
	 * @param connection
	 *            {@link Socket} to add
	 */
	public void addConnection(Socket connection) {
		synchronized (this.streamConnections) {
			int nextPtr = this.lastAddedPtr + 1;
			if (nextPtr == this.streamConnections.length) {
				nextPtr = 0;
			}

			if (nextPtr == this.lastReadPtr) {
				HokaLogger.instance.error("too many connections max is:" + this.maxOpenedConnections); //$NON-NLS-1$
				tooManyOpenConnections(connection);
				return;
			}

			String host = DEFAULT_ADDRESS;
			final InetAddress address = connection.getInetAddress();
			if (address != null) {
				host = address.getHostAddress();
			}
			HokaLogger.instance.trace(connection.hashCode() + TAB + host + "\t connection received"); //$NON-NLS-1$

			this.streamConnections[nextPtr] = connection;
			this.lastAddedPtr = nextPtr;
			this.streamConnections.notify();
		}
	}

	/**
	 * Get the next {@link Socket} to process. Block until a new connection is available or server is stopped.
	 *
	 * @return null if server is stopped
	 */
	@Nullable
	public Socket getNextStreamConnection() {
		synchronized (this.streamConnections) {
			while (this.lastAddedPtr == this.lastReadPtr) {
				if (isStopped()) {
					return null;
				}
				try {
					this.streamConnections.wait();
				} catch (final InterruptedException e) {
					HokaLogger.instance.error(e);
					Thread.currentThread().interrupt();
				}
			}

			int nextPtr = this.lastReadPtr + 1;
			if (nextPtr == this.streamConnections.length) {
				nextPtr = 0;
			}
			Socket connection = this.streamConnections[nextPtr];
			this.lastReadPtr = nextPtr;
			// allow GC
			this.streamConnections[nextPtr] = null;
			return connection;
		}
	}

	/**
	 * Returns {@code true} if the {@link TcpServer} is stopped.
	 *
	 * @return {@code true} if the {@link TcpServer} is stopped, {@code false} otherwise
	 */
	public boolean isStopped() {
		return this.serverSocket == null || this.thread == null;
	}

	/**
	 * Returns the name of this TCPServer.
	 *
	 * @return the string "TCPServer"
	 */
	protected String getName() {
		return TcpServer.class.getSimpleName();
	}

	/**
	 * Called when a connection cannot be added to the buffer. By default, the connection is closed.
	 *
	 * @param connection
	 *            {@link Socket} that can not be added
	 */
	protected void tooManyOpenConnections(Socket connection) {
		try {
			connection.close();
		} catch (IOException e) {
			HokaLogger.instance.error(e);
		}
	}

	/**
	 * Returns a new Server process as {@link Runnable}.
	 *
	 * @return a new Server process as {@link Runnable}
	 */
	private Runnable newProcess() {
		return new Runnable() {
			@Override
			public void run() {
				boolean running = true;
				while (running) {
					if (isStopped()) {
						break;
					}
					try {
						ServerSocket serverSocket = TcpServer.this.serverSocket;
						if (serverSocket != null) {
							Socket connection = serverSocket.accept();
							connection.setSoTimeout(TcpServer.this.timeout);
							addConnection(connection);
						}
					} catch (IOException e) {
						if (isStopped()) {
							running = false;
						}

						// Connection cannot be handled but server is still alive.
						// It may happen if too many requests are made simultaneously.
						if (running) {
							HokaLogger.instance.error(e);
						}
					}
				}
			}
		};
	}

	/**
	 * Gets the port number to which this server is listening to, if any.
	 *
	 * @return the port number to which this server is listening or-1 if the socket is not bound yet.
	 */
	public int getPort() {
		return this.serverSocket != null ? this.serverSocket.getLocalPort() : -1;
	}

	/**
	 * Gets the address to which this server is bound, if any.
	 *
	 * @return the address to which this server is bound,or null if the socket is unbound.
	 */
	@Nullable
	public InetAddress getInetAddress() {
		return this.serverSocket != null ? this.serverSocket.getInetAddress() : null;
	}

}
