/*
 * Java
 *
 * Copyright 2021 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.hoka.http;

import static ej.hoka.http.HttpConstants.EMPTY;
import static ej.hoka.http.HttpConstants.FSLASH;
import static ej.hoka.http.HttpConstants.HEADER_ACCEPT;
import static ej.hoka.http.HttpConstants.WILDCARD;
import static ej.hoka.http.HttpRequest.AFTER;
import static ej.hoka.http.HttpRequest.AFTER_ALL;
import static ej.hoka.http.HttpRequest.BEFORE;
import static ej.hoka.http.HttpRequest.BEFORE_ALL;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;

import ej.annotation.Nullable;
import ej.basictool.ArrayTools;
import ej.basictool.map.PackedMap;
import ej.hoka.http.requesthandler.RequestHandler;
import ej.hoka.http.requesthandler.StaticFilesHandler;
import ej.hoka.log.HokaLogger;

/**
 * A request handler that exposes a REST API. Handles GET, POST, PUT and DELETE operations on endpoints.
 * <p>
 * The endpoint that handles the request is the endpoint with the most specific path that matches the request.
 */
class RouteHandler implements RequestHandler {

	private static final String PATH_PARAM_PREFIX = ":"; //$NON-NLS-1$
	protected static final String ALL_PATH = "+/*all-routes";//$NON-NLS-1$

	private Route[] routes = new Route[0];
	private Route[] globalFilters = new Route[0];
	private Route[] pathFilters = new Route[0];
	private final PackedMap<Class<? extends Exception>, RequestHandler> exceptionHandlers = new PackedMap<>();
	private final boolean trailingSlashSupport;
	@Nullable
	private final String routeBase;
	@Nullable
	private final StaticFilesHandler staticFilesHandler;

	RouteHandler(String apiBase) {
		this(apiBase, null, false);
	}

	RouteHandler(String apiBase, @Nullable final StaticFilesHandler staticFilesHandler) {
		this(apiBase, staticFilesHandler, false);
	}

	RouteHandler(String apiBase, @Nullable final StaticFilesHandler staticFilesHandler,
			final boolean trailingSlashSupport) {
		this.routeBase = apiBase;
		this.staticFilesHandler = staticFilesHandler;
		this.trailingSlashSupport = trailingSlashSupport;
	}

	/**
	 * Processes the given HTTP request with the request handlers of every route from the given Route array whose HTTP
	 * method fits the given method.
	 *
	 * @param routes
	 *            The array of Routes to process the request with.
	 * @param HttpMethod
	 *            The HTTP method to which routes have to have for the request to be processed with.
	 * @param request
	 *            The HTTP request to be processed.
	 * @param response
	 *            The HTTP response to be sent.
	 */
	private void processRoutesWithFilter(Route[] routes, int HttpMethod, HttpRequest request, HttpResponse response) {
		for (Route r : routes) {
			if (HttpMethod == r.getHttpMethod()) {
				r.getHandler().process(request, response);
			}
		}
	}

	/**
	 * Processes the given HTTP request with the request handlers of every route from the given Route list whose HTTP
	 * method fits the given method.
	 *
	 * @param routes
	 *            The list of Routes to process the request with.
	 * @param HttpMethod
	 *            The HTTP method to which routes have to have for the request to be processed with.
	 * @param request
	 *            The HTTP request to be processed.
	 * @param response
	 *            The HTTP response to be sent.
	 */
	private void processRoutesWithFilter(List<Route> routes, int HttpMethod, HttpRequest request,
			HttpResponse response) {
		for (Route r : routes) {
			if (HttpMethod == r.getHttpMethod()) {
				r.getHandler().process(request, response);
			}
		}
	}

	@Override
	public void process(HttpRequest request, HttpResponse response) {

		final int requestHttpMethod = request.getMethod();
		final String requestURI = request.getURI();
		final String acceptType = request.getHeader(HEADER_ACCEPT);

		// static file serving
		StaticFilesHandler staticFilesHandler = this.staticFilesHandler;
		if (staticFilesHandler != null) {
			Map<String, String> headers = new HashMap<>();
			InputStream is = staticFilesHandler.serve(requestURI, headers);
			if (is != null) {
				response.setData(is);
				response.addHeaders(headers);
				return;
			}
		}

		final List<Route> allowedRoutes = matchRoute(requestHttpMethod, acceptType, requestURI);
		Route route = null;
		for (Route r : allowedRoutes) {
			if (requestHttpMethod == r.getHttpMethod()) { // Get first match route.
				route = r;
				break;
			}
		}

		// Populate path and splat parameters for this request if a route was found
		// we don't return a not found error at this point to let global before filter handle the request
		// if none of the filters halt the request and the route is null a RouteNotFoundException is returned
		if (route != null) {
			setPathAndSplatParams(route.getPath(), request);
		}

		try {
			// Processing flow
			// before all
			processRoutesWithFilter(this.globalFilters, BEFORE_ALL, request, response);

			final List<Route> beforeAfterpathFilters = matchPathFilters(requestHttpMethod, acceptType, requestURI);
			// before filter
			processRoutesWithFilter(beforeAfterpathFilters, BEFORE, request, response);

			// main route
			if (route != null) {
				route.getHandler().process(request, response);
			} else if (!allowedRoutes.isEmpty()) {
				throw new MethodNotAllowedException(allowedRoutes);
			} else {
				throw new RouteNotFoundException();
			}

			// after filter
			processRoutesWithFilter(beforeAfterpathFilters, AFTER, request, response);

			// after all filter
			processRoutesWithFilter(this.globalFilters, AFTER_ALL, request, response);
		} catch (final HaltException e) {
			throw e; // prevent catching this in a request handler
		} catch (final Exception e) {
			if (this.exceptionHandlers.containsKey(e.getClass())) {
				RequestHandler key = this.exceptionHandlers.get(e.getClass());
				assert key != null; // ensured in the condition
				key.process(request, response);
			} else {
				throw e;
			}
		}
	}

	/**
	 * Populate splat and path parameters for every request
	 *
	 * @param routeURI
	 *            endpoint path containing splat of path params
	 * @param request
	 *            the request to populate with parameters
	 */
	protected void setPathAndSplatParams(final String routeURI, final HttpRequest request) {
		if (!routeURI.contains(PATH_PARAM_PREFIX) && !routeURI.contains(WILDCARD)) {
			return;
		}

		final String[] routePathParts = pathToList(routeURI);
		final String[] requestPathParts = pathToList(request.getURI());

		int nbrOfEndpointParts = routePathParts.length;
		int nbrOfRequestParts = requestPathParts.length;

		final Map<String, String> pathParams = new HashMap<>();
		final List<String> splatParams = new ArrayList<>();

		for (int i = 0; i < nbrOfEndpointParts && i < nbrOfRequestParts; i++) {
			if (isPathParam(routePathParts[i])) {
				pathParams.put(routePathParts[i].toLowerCase(), requestPathParts[i]);
			} else if (isSplatParam(routePathParts[i])) {
				StringBuilder splatParam = new StringBuilder(requestPathParts[i]);
				if (nbrOfEndpointParts != nbrOfRequestParts && (i == nbrOfEndpointParts - 1)) {
					// if last endpoint part. add the rest of the query path to the splat param value
					for (int j = i + 1; j < nbrOfRequestParts; j++) {
						splatParam.append(FSLASH);
						splatParam.append(requestPathParts[j]);
					}
				}
				splatParams.add(splatParam.toString());
			}
		}

		request.addPathParameters(pathParams);
		request.addSplatParameters(splatParams);
	}

	/**
	 *
	 * @param part
	 *            path part
	 * @return true if it's a path param, false otherwise
	 */
	protected boolean isPathParam(final String part) {
		return part.startsWith(PATH_PARAM_PREFIX);
	}

	/**
	 *
	 * @param part
	 *            path part
	 * @return true if it's a splat param, false otherwise
	 */
	protected boolean isSplatParam(final String part) {
		return part.equals(WILDCARD);
	}

	/**
	 * @param path
	 *            path to split
	 * @return path part after split on '/'
	 */
	protected String[] pathToList(final String path) {
		if (FSLASH.equals(path)) {
			return new String[] { EMPTY };
		}

		final StringTokenizer pathTokenizer = new StringTokenizer(path, FSLASH);
		String[] pathParts = new String[0];
		while (pathTokenizer.hasMoreTokens()) {
			final String pathPart = pathTokenizer.nextToken();
			pathParts = ArrayTools.add(pathParts, pathPart);
		}

		if (this.trailingSlashSupport && path.endsWith(FSLASH)) {
			// paths 'host/hello' 'host/hello/' doesn't match in this case
			pathParts = ArrayTools.add(pathParts, EMPTY);
		}

		return pathParts;
	}

	/**
	 * Find the route that matches the request path.
	 *
	 * @param httpMethod
	 *            http method
	 * @param acceptType
	 *            accepted type
	 * @param requestPath
	 *            the HTTP Request path to match.
	 *
	 * @return the {@link Route} the first route that matches the request path. or a list of possible routes with other
	 *         allowed methods if not
	 */
	protected List<Route> matchRoute(final int httpMethod, @Nullable String acceptType, String requestPath) {
		return matchRoute(this.routes, httpMethod, acceptType, requestPath);
	}

	/**
	 * Find the path filter that matches the request path.
	 *
	 * @param httpMethod
	 *            http method
	 * @param acceptType
	 *            accepted type
	 * @param requestPath
	 *            the HTTP Request path to match.
	 *
	 * @return the {@link Route} the first route that matches the request path. or a list of possible routes with other
	 *         allowed methods if not
	 */
	protected List<Route> matchPathFilters(final int httpMethod, @Nullable String acceptType, String requestPath) {
		return matchRoute(this.pathFilters, httpMethod, acceptType, requestPath);
	}

	private List<Route> matchRoute(final Route[] routes, final int httpMethod, @Nullable String acceptType,
			String requestPath) {
		if (!this.trailingSlashSupport && !FSLASH.equals(requestPath) && requestPath.endsWith(FSLASH)) {
			requestPath = requestPath.substring(0, requestPath.length() - 1);
		}

		String[] requestAcceptedTypes = null;
		final String[] requestPathParts = pathToList(requestPath);
		List<Route> matches = new ArrayList<>();
		for (final Route route : routes) {
			if (requestAcceptedTypes == null && !route.acceptAllContentTypes()) {
				requestAcceptedTypes = getRequestAcceptedTypes(acceptType); // calculate request accepted types
			}

			if (route.getPath().equals(requestPath) && matchContentType(requestAcceptedTypes, route)) {
				matches.add(route); // Exact path matching
				if (httpMethod == route.getHttpMethod()) {
					return matches;
				}
			}

			// route with path or splat params
			if ((route.getPath().contains(WILDCARD) || route.getPath().contains(PATH_PARAM_PREFIX))
					&& matchPath(route.getPath(), pathToList(route.getPath()), requestPathParts)
					&& matchContentType(requestAcceptedTypes, route)) {
				matches.add(route);
				if (httpMethod == route.getHttpMethod()) {
					return matches;
				}
			}
		}

		return matches;
	}

	/**
	 * @param requestAcceptedTypes
	 *            list of request accepted content types.
	 * @param route
	 *            registered route
	 * @return true if the registered route support one of the request accepted content types.
	 */
	protected boolean matchContentType(@Nullable String[] requestAcceptedTypes, final Route route) {
		return route.acceptAllContentTypes() || requestAcceptType(requestAcceptedTypes, route.getAcceptType());
	}

	/**
	 * @param requestAcceptedTypes
	 *            list of accepted types
	 * @param routeAcceptType
	 *            type that route support
	 * @return true if it's a supported type, false otherwise
	 */
	protected boolean requestAcceptType(@Nullable String[] requestAcceptedTypes,
			@Nullable final String routeAcceptType) {
		return requestAcceptedTypes == null || requestAcceptedTypes.length == 0
				|| ArrayTools.containsEquals(requestAcceptedTypes, Route.DEFAULT_ACCEPT_TYPE) || routeAcceptType == null
				|| ArrayTools.containsEquals(requestAcceptedTypes, routeAcceptType);
	}

	/**
	 * @param acceptType
	 *            value of request header Accept
	 * @return list of accepted type extracted from the request
	 */
	protected String[] getRequestAcceptedTypes(@Nullable String acceptType) {
		if (acceptType == null || acceptType.isEmpty()) {
			return new String[] { Route.DEFAULT_ACCEPT_TYPE };
		}

		final StringTokenizer acceptTypeParser = new StringTokenizer(acceptType, ","); //$NON-NLS-1$
		String[] acceptedTypes = new String[0];
		while (acceptTypeParser.hasMoreTokens()) {
			String type = acceptTypeParser.nextToken().trim();
			if (type.contains(";")) { //$NON-NLS-1$
				// remove other parts from MIME type, mostly to remove q which is not supported by HOKA
				type = type.substring(0, type.indexOf(';'));
			}
			acceptedTypes = ArrayTools.add(acceptedTypes, type);
		}

		return acceptedTypes;
	}

	/**
	 * * Check if two paths are matching
	 *
	 * @param routeURI
	 *            registered endpoint
	 * @param routePathParts
	 *            part of the registered endpoint to be compared
	 * @param requestPathParts
	 *            part of the request path
	 * @return true if the two path matches, false otherwise
	 */
	protected boolean matchPath(String routeURI, String[] routePathParts, String[] requestPathParts) {
		if (requestPathParts.length < routePathParts.length) {
			return false; // request path parts can't be handled by smaller registered path
		}
		if (routePathParts.length != requestPathParts.length && !routeURI.endsWith(WILDCARD)) {
			return false;
		}

		for (int i = 0; i < routePathParts.length; i++) {
			String routePathPart = routePathParts[i];

			if ((i == routePathParts.length - 1) && (routePathPart.equals(WILDCARD) && routeURI.endsWith(WILDCARD))) {
				// last element is a wildcard and endpoints ends with a wildcard
				return true;
			}

			String requestPathPart = requestPathParts[i];
			if ((!routePathPart.startsWith(PATH_PARAM_PREFIX)) && !routePathPart.equalsIgnoreCase(requestPathPart)
					&& !routePathPart.equals(WILDCARD)) {
				// current part is not a path param and parts are not the same and endpoint is not a splat param
				return false;
			}
		}

		return true;
	}

	/**
	 * add a new route
	 *
	 * @param httpMethod
	 *            HTTP method {@link HttpRequest#GET} {@link HttpRequest#POST} {@link HttpRequest#DELETE}
	 *            {@link HttpRequest#PUT}
	 * @param path
	 *            request path
	 *
	 * @param handler
	 *            request handler
	 */
	protected void add(int httpMethod, String path, RequestHandler handler) {
		add(httpMethod, path, null, handler);
	}

	/**
	 * add a new route
	 *
	 * @param httpMethod
	 *            HTTP method {@link HttpRequest#GET} {@link HttpRequest#POST} {@link HttpRequest#DELETE}
	 *            {@link HttpRequest#PUT}
	 * @param path
	 *            request path
	 * @param acceptType
	 *            accepted type by this route
	 * @param handler
	 *            request handler
	 */
	protected void add(int httpMethod, @Nullable String path, @Nullable String acceptType,
			@Nullable RequestHandler handler) {
		if (handler == null || path == null || path.isEmpty() || path.trim().isEmpty()) {
			throw new IllegalArgumentException();
		}
		final Route newRoute = new Route(httpMethod, appendBase(path), acceptType, handler);
		switch (httpMethod) {
		case HttpRequest.AFTER_ALL:
		case HttpRequest.BEFORE_ALL:
			this.globalFilters = ArrayTools.add(this.globalFilters, newRoute);
			break;
		case HttpRequest.BEFORE:
		case HttpRequest.AFTER:
			if (ArrayTools.containsEquals(this.pathFilters, newRoute)) {
				throw new IllegalArgumentException(
						newRoute.getHttpMethodAsString() + " | " + newRoute.getPath() + " already exists."); //$NON-NLS-1$ //$NON-NLS-2$ // NOSONAR
			}
			this.pathFilters = ArrayTools.add(this.pathFilters, newRoute);
			break;
		case HttpRequest.GET:
		case HttpRequest.POST:
		case HttpRequest.PUT:
		case HttpRequest.DELETE:
		case HttpRequest.HEAD:
		case HttpRequest.CONNECT:
		case HttpRequest.OPTIONS:
		case HttpRequest.TRACE:
		case HttpRequest.PATCH:
			if (ArrayTools.containsEquals(this.routes, newRoute)) {
				throw new IllegalArgumentException(
						newRoute.getHttpMethodAsString() + " | " + newRoute.getPath() + " already exists."); //$NON-NLS-1$ //$NON-NLS-2$ // NOSONAR
			}
			this.routes = ArrayTools.add(this.routes, newRoute);
			break;
		default:
			throw new IllegalArgumentException("Http method not supported"); //$NON-NLS-1$
		}
		logRegistredRoute(newRoute);
	}

	private void logRegistredRoute(Route r) {
		HokaLogger.instance.info(r.getHttpMethodAsString() + "   \t" + r.getPath() + "\t\t\t" + //$NON-NLS-1$ //$NON-NLS-2$
				r.getHandler().getClass().getName());
	}

	protected void addExceptionHandler(@Nullable Class<? extends Exception> clazz, @Nullable RequestHandler handler) {
		if (clazz == null || handler == null) {
			throw new IllegalArgumentException();
		}
		if (this.exceptionHandlers.containsKey(clazz)) {
			RequestHandler value = this.exceptionHandlers.get(clazz);
			assert value != null; // keys and values ensured non null at registration
			throw new IllegalArgumentException("'" + clazz + "' is already mapped to '" //$NON-NLS-1$ //$NON-NLS-2$ // NOSONAR
					+ value.getClass().getName() + "'"); //$NON-NLS-1$ // NOSONAR
		}
		this.exceptionHandlers.put(clazz, handler);
	}

	protected List<Route> getRoutes() {
		return Collections.unmodifiableList(Arrays.asList(this.routes));
	}

	private String appendBase(String path) {
		if (this.routeBase == null || EMPTY.equals(this.routeBase) || path.startsWith(FSLASH)) {
			if (!path.startsWith(FSLASH)) {
				return FSLASH + path;
			}
			return path;
		}

		return this.routeBase + path;
	}

}
