/*
 * Copyright 2023-2024 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.fp.widget.display.brs;

import ej.fp.widget.display.DisplayBufferManager;
import ej.microui.display.LLUIDisplayImpl;
import ej.microui.display.Rectangle;

/**
 * The buffer refresh strategy (or BRS) is responsible of refreshing the display and restoring the data of the front
 * buffer to the back buffer.
 */
public interface BufferRefreshStrategy {

	/**
	 * Sets the number of buffers.
	 *
	 * @param bufferCount
	 *            the number of buffers.
	 */
	void setBufferCount(int bufferCount);

	/**
	 * Sets the display size.
	 *
	 * @param width
	 *            the width of the display.
	 * @param height
	 *            the height of the display.
	 */
	void setDisplaySize(int width, int height);

	/**
	 * Notifies that a region will be modified by the application in the display back buffer.
	 * <p>
	 * The implementation can perform some drawings.
	 * <p>
	 * Called when a drawing is done or when the user declares a dirty region.
	 *
	 * @param bufferManager
	 *            the display buffer manager.
	 * @param rectangle
	 *            the drawing region.
	 * @see LLUIDisplayImpl#newDrawingRegion(int, int, int, int, boolean)
	 * @param drawingNow
	 *            <code>true</code> if a drawing is following this call, <code>false</code> otherwise.
	 */
	void newDrawingRegion(DisplayBufferManager bufferManager, Rectangle rectangle, boolean drawingNow);

	/**
	 * Refreshes the display by calling {@link DisplayBufferManager#flush(ej.fp.widget.display.Rectangle[])}.
	 *
	 * @param bufferManager
	 *            the display buffer manager.
	 */
	void refresh(DisplayBufferManager bufferManager);
}
