/*
 * Java
 *
 * Copyright 2009-2025 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.fp.widget;

import java.awt.Color;
import java.awt.Container;
import java.awt.Desktop;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.nio.file.Paths;
import java.text.MessageFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.concurrent.atomic.AtomicInteger;

import javax.imageio.ImageIO;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.ImageIcon;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JSeparator;
import javax.swing.JToolBar;
import javax.swing.SwingConstants;

import ej.fp.Composite;
import ej.fp.FrontPanel;
import ej.fp.Image;
import ej.fp.MouseListener;
import ej.fp.Widget.WidgetAttribute;
import ej.fp.Widget.WidgetDescription;
import ej.fp.widget.display.DisplayBufferManager;
import ej.fp.widget.display.brs.BufferRefreshStrategy;
import ej.fp.widget.display.brs.PredrawRefreshStrategy;
import ej.fp.widget.display.buffer.DisplayBufferPolicy;
import ej.fp.widget.display.buffer.SwapDoubleBufferPolicy;
import ej.microui.display.LLUIDisplayImpl;
import ej.microui.display.MicroUIImageFormat;

/**
 * This widget is an implementation of MicroUI {@link LLUIDisplayImpl} interface. An implementation of this interface is
 * required to be able to use a MicroUI Display in a MicroEJ application.
 * <p>
 * This widget has been implemented to target a standard display: a display whose pixel format is
 * {@link MicroUIImageFormat#MICROUI_IMAGE_FORMAT_ARGB8888} , {@link MicroUIImageFormat#MICROUI_IMAGE_FORMAT_RGB888},
 * {@link MicroUIImageFormat#MICROUI_IMAGE_FORMAT_RGB565}, {@link MicroUIImageFormat#MICROUI_IMAGE_FORMAT_ARGB1555},
 * {@link MicroUIImageFormat#MICROUI_IMAGE_FORMAT_ARGB4444}, {@link MicroUIImageFormat#MICROUI_IMAGE_FORMAT_C4},
 * {@link MicroUIImageFormat#MICROUI_IMAGE_FORMAT_C2} or {@link MicroUIImageFormat#MICROUI_IMAGE_FORMAT_C1}. To avoid to
 * create a subclass of this class when the pixel format is custom, an extension of this widget can be set in front
 * panel description file (fp file) using the widget attribute <code>extensionClass</code>.
 * <p>
 * This widget manages the double buffering mode (see {@link LLUIDisplayImpl#isDoubleBuffered()}. However all involved
 * methods by this notion are implemented according the value returned by {@link #isDoubleBuffered()}. The optional
 * widget attribute <code>doubleBufferFeature</code> can be configured to override default behavior (enabled by
 * default).
 * <p>
 * When double buffering mode is enabled, this widget uses a third-party thread to simulate the asynchronous flush
 * action. This allows to have the same behavior than embedded side: the next drawing after a flush has to wait a while
 * before be performed. The flush action takes two parameters in consideration: the time to perform the flush itself (==
 * the time to transmit data to the embedded display) and the embedded display's refresh rate. These times are useless
 * when the double buffering mode is disabled.
 * <p>
 * This widget manages a backlight with dimming (0 to 100%) (see {@link LLUIDisplayImpl#hasBacklight()}. However all
 * involved methods by this notion are implemented according the value returned by {@link #hasBacklight()}. The optional
 * widget attribute <code>backlightFeature</code> can be configured to override default behavior (enabled by default).
 * <p>
 * The filter image is used to define the display rendering area. Outside this area, the display buffer data is not
 * drawn. This image must have the same size (@see {@link #setWidth(int)} and {@link #setHeight(int)}) than the widget
 * itself. If not, an error is thrown in {@link #finalizeConfiguration()}.
 */
@SuppressWarnings("nls")
@WidgetDescription(attributes = { @WidgetAttribute(name = "label", isOptional = true), @WidgetAttribute(name = "x"),
		@WidgetAttribute(name = "y"), @WidgetAttribute(name = "width"), @WidgetAttribute(name = "height"),
		@WidgetAttribute(name = "displayWidth", isOptional = true),
		@WidgetAttribute(name = "displayHeight", isOptional = true),
		@WidgetAttribute(name = "initialColor", isOptional = true), @WidgetAttribute(name = "alpha", isOptional = true),
		@WidgetAttribute(name = "flushTime", isOptional = true),
		@WidgetAttribute(name = "refreshRate", isOptional = true),
		@WidgetAttribute(name = "backlightFeature", isOptional = true),
		@WidgetAttribute(name = "filter", isOptional = true),
		@WidgetAttribute(name = "extensionClass", isOptional = true),
		@WidgetAttribute(name = "bufferPolicyClass", isOptional = true),
		@WidgetAttribute(name = "refreshStrategyClass", isOptional = true) })
public class Display extends Composite implements LLUIDisplayImpl, MouseListener {

	private static final String IMAGES_PREFIX = "images/"; //$NON-NLS-1$

	private static final String DISPLAY_COORDINATES_PATTERN = "Display: {0}, {1}";
	private static final int TOOL_BAR_POSITION = 1;
	private static final int STATUS_BAR_POSITION = 2;
	private static final int STATUS_BAR_LEFT_POSITION = 0;
	private static final int STATUS_BAR_CENTER_POSITION = 1;
	private static final int SCREENSHOT_NOTIFICATION_DURATION = 5000;

	/**
	 * This interface is a subset of {@link LLUIDisplayImpl} interface (provided by MicroUI graphical engine and
	 * required to use a MicroUI Display in a MicroEJ application).
	 * <p>
	 * It provides a set of minimal methods to customize the {@link Display} widget (which implements
	 * {@link LLUIDisplayImpl}) without needing to subclass it (no need to create a sub-widget "display" with same
	 * characteritics and same {@link WidgetAttribute}, just to override one setting).
	 * <p>
	 * An implementation classname of this interface can be set thanks the widget attribute
	 * {@link Display#setExtensionClass(String)}.
	 */
	public static interface DisplayExtension {

		/**
		 * Asks if the display is a colored display or not.
		 *
		 * @param display
		 *            the display widget.
		 *
		 * @return true when the display is not a grayscale display, false otherwise.
		 * @see LLUIDisplayImpl#isColor()
		 */
		boolean isColor(Display display);

		/**
		 * Gets the number of colors that can be represented on the device.
		 *
		 * @param display
		 *            the display widget.
		 *
		 * @return the number of colors of the display.
		 * @see LLUIDisplayImpl#getNumberOfColors()
		 */
		int getNumberOfColors(Display display);

		/**
		 * Converts the 32-bit ARGB color format (A-R-G-B) into the display color format.
		 *
		 * @param display
		 *            the display widget.
		 * @param argbColor
		 *            the color to convert.
		 * @return the display color equivalent to <code>microUIColor</code>.
		 * @see LLUIDisplayImpl#convertARGBColorToDisplayColor(int)
		 */
		int convertARGBColorToDisplayColor(Display display, int argbColor);

		/**
		 * Converts the display color format into a 32-bit ARGB color format (A-R-G-B).
		 *
		 * @param display
		 *            the display widget.
		 * @param displayColor
		 *            the color to convert.
		 * @return the MicroUI color equivalent to <code>displayColor</code>.
		 * @see LLUIDisplayImpl#convertDisplayColorToARGBColor(int)
		 */
		int convertDisplayColorToARGBColor(Display display, int displayColor);

		/**
		 * Prepares the blending of two ARGB colors (only useful when the LCD is a palletized LCD).
		 * <p>
		 * By default this feature is not used.
		 *
		 * @param display
		 *            the display widget.
		 * @param foreground
		 *            the foreground ARGB color to convert.
		 * @param background
		 *            the background ARGB color to convert.
		 * @return true when the indexes have been found, false otherwise.
		 * @see LLUIDisplayImpl#prepareBlendingOfIndexedColors(AtomicInteger, AtomicInteger)
		 */
		default boolean prepareBlendingOfIndexedColors(Display display, AtomicInteger foreground,
				AtomicInteger background) {
			return false;
		}
	}

	/**
	 * Property to change the platform's flush time.
	 * <p>
	 * Add the property <code>-Dej.fp.widget.display.flushTime=xxx</code> to override the flush time set (or not) in the
	 * front panel file. A negative or zero value disables the simulation of the flush time.
	 */
	private static final String PROPERTY_FLUSH_TIME = "ej.fp.display.flushTime";
	private static final String PROPERTY_FLUSH_TIME_LEGACY = "ej.fp.widget.display.flushTime";

	/**
	 * Property to change the display's refresh rate.
	 * <p>
	 * Add the property <code>-Dej.fp.widget.display.refreshRate=xxx</code> to override the refresh rate set (or not) in
	 * the front panel file. A negative or zero value disables the simulation of the refresh rate.
	 */
	private static final String PROPERTY_REFRESH_RATE = "ej.fp.display.refreshRate";
	private static final String PROPERTY_REFRESH_RATE_LEGACY = "ej.fp.widget.display.refreshRate";

	private DisplayBufferManager bufferManager;

	/**
	 * Buffer visible on front panel viewer: contains application drawings (see {@link #bufferManager}) and some post
	 * transformations like mask reduction, backlight and contrast.
	 */
	private Image visibleBuffer;

	// widget extension which describes the display.
	private DisplayExtension extension; // null init
	private String extensionClassName;

	// display size in pixels, may be different than widget size (see setDisplaylWidth())
	private int displayWidth; // 0 init
	private int displayHeight; // 0 init

	private String bufferPolicyClass;
	private String refreshStrategyClass;

	// display characteristics
	private int initialColor; // 0 init
	private byte alpha;
	private boolean hasBacklight;
	private int flushTimeMs;
	private int refreshRateHz;
	private int refreshRateTimeMs;

	// current backlight
	private int backlight;

	private AbstractAction screenshotAction;
	private JLabel coordinatesLabel;
	private JLabel screenshotNotificationLabel;
	private int screenshotNotificationCounter;
	private File outputDir;

	/**
	 * Creates the widget display
	 */
	public Display() {
		// fully opaque by default
		this.alpha = (byte) 0xff;

		// backlight feature enabled by default
		this.hasBacklight = true;
	}

	/**
	 * Sets the width of the display in pixels (see class comment).
	 * <p>
	 * This method should only be called by front panel parser.
	 *
	 * @param width
	 *            the width to set.
	 */
	public void setDisplayWidth(int width) {
		if (width < 0) {
			throw new IllegalArgumentException("Display width cannot be negative.");
		}
		this.displayWidth = width;
	}

	/**
	 * Sets the height of the display in pixels (see class comment).
	 * <p>
	 * This method should only be called by front panel parser.
	 *
	 * @param height
	 *            the height to set.
	 */
	public void setDisplayHeight(int height) {
		if (height < 0) {
			throw new IllegalArgumentException("Display height cannot be negative.");
		}
		this.displayHeight = height;
	}

	/**
	 * Gets the display width.
	 *
	 * @return the display width.
	 */
	public int getDisplayWidth() {
		return this.displayWidth;
	}

	/**
	 * Gets the display height.
	 *
	 * @return the display height.
	 */
	public int getDisplayHeight() {
		return this.displayHeight;
	}

	/**
	 * Sets the initial color of the display (color used on startup, just before first drawing). By default the initial
	 * color is 0x0 (black).
	 * <p>
	 * This is a RGB color (24 bits: 8-8-8). Alpha level is ignored.
	 * <p>
	 * This method should only be called by front panel parser.
	 *
	 * @param initialColor
	 *            the initial color to set.
	 */
	public void setInitialColor(int initialColor) {
		this.initialColor = initialColor | 0xff000000;
	}

	/**
	 * Sets the opacity level (alpha) of the display in order to obtain a translucent display. By default the display is
	 * fully opaque.
	 * <p>
	 * This method should only be called by front panel parser.
	 *
	 * @param alpha
	 *            the opacity level to set, 0x00 is fully transparent and 0xff fully opaque.
	 * @throws IllegalArgumentException
	 *             if the given argument is not between 0x0 and 0xff.
	 */
	public void setAlpha(int alpha) {
		if (alpha < 0x0 || alpha > 0xff) {
			throw new IllegalArgumentException("The opacity level must be a value between 0x0 and 0xff");
		}
		this.alpha = (byte) alpha;
	}

	/**
	 * Simulates the embedded display's flush time when double buffering mode is enabled.
	 * <p>
	 * The widget will wait this time before unlocking the caller to {@link #waitFlush()}.
	 * <p>
	 * The time is expressed in milliseconds. A negative or zero value (default value) indicates the flush time is
	 * instantaneous. Typical values are between 5 and 10ms.
	 *
	 * @param ms
	 *            the flush time in milliseconds.
	 */
	public void setFlushTime(int ms) {
		this.flushTimeMs = ms > 0 ? ms : 0;
	}

	/**
	 * Simulates the embedded display's refresh rate when double buffering mode is enabled.
	 * <p>
	 * When set, the widget is only allowed to initiate a flush action at every refresh tick. It is synchronized on the
	 * hardware display's periodic signal (tearing).
	 * <p>
	 * The signal is expressed in hertz. A negative or zero value (default value) indicates the display does not manage
	 * the display's refresh rate and the flush action can start at once. Typical value is 60Hz.
	 *
	 * @param hertz
	 *            the refresh rate in hertz.
	 */
	public void setRefreshRate(int hertz) {
		if (hertz > 0) {
			this.refreshRateHz = hertz > 0 ? hertz : 0;
			this.refreshRateTimeMs = 1000 / hertz;
		} else {
			this.refreshRateHz = 0;
			this.refreshRateTimeMs = 0;
		}
	}

	/**
	 * Enables or disables the backlight feature.
	 * <p>
	 * By default the backlight feature is enabled.
	 * <p>
	 * See {@link LLUIDisplayImpl#hasBacklight()} to have more information.
	 *
	 * @param enable
	 *            true to enable backlight feature (default value).
	 */
	public void setBacklightFeature(boolean enable) {
		this.hasBacklight = enable;
	}

	/**
	 * Defines the optional class name of the display extension. This allows to customize this display widget without
	 * creating a sub-class of {@link Display}.
	 * <p>
	 * When set, a check is performed in {@link #finalizeConfiguration()} to verify whether this class is valid.
	 * <p>
	 * This method should only be called by front panel parser.
	 *
	 * @param extensionClassName
	 *            the user display extension class name.
	 */
	public void setExtensionClass(String extensionClassName) {
		this.extensionClassName = extensionClassName;
	}

	/**
	 * Sets the buffer policy.
	 *
	 * @param bufferPolicyClassName
	 *            the buffer policy class name to set.
	 */
	public void setBufferPolicyClass(String bufferPolicyClassName) {
		this.bufferPolicyClass = bufferPolicyClassName;
	}

	/**
	 * Sets the refresh strategy.
	 *
	 * @param refreshStrategyClassName
	 *            the refresh strategy class name to set.
	 */
	public void setRefreshStrategyClass(String refreshStrategyClassName) {
		this.refreshStrategyClass = refreshStrategyClassName;
	}

	@Override
	public synchronized void finalizeConfiguration() {
		super.finalizeConfiguration();

		// display size may be different than widget size
		if (this.displayWidth == 0) {
			this.displayWidth = getWidth();
		}
		if (this.displayHeight == 0) {
			this.displayHeight = getHeight();
		}

		if (this.extensionClassName != null) {
			// check user class
			FrontPanel.getFrontPanel().verifyUserClass(getClass(), DisplayExtension.class, this.extensionClassName);
		}
		if (this.bufferPolicyClass != null) {
			// check user class
			FrontPanel.getFrontPanel().verifyUserClass(getClass(), DisplayBufferPolicy.class, this.bufferPolicyClass);
		}
		if (this.refreshStrategyClass != null) {
			// check user class
			FrontPanel.getFrontPanel().verifyUserClass(getClass(), BufferRefreshStrategy.class,
					this.refreshStrategyClass);
		}

		// no need to repaint behind display
		setOverlay(true);

		// flush time and refresh rate may have been customized by the application
		setFlushTime(getValue(PROPERTY_FLUSH_TIME, PROPERTY_FLUSH_TIME_LEGACY, this.flushTimeMs));
		setRefreshRate(getValue(PROPERTY_REFRESH_RATE, PROPERTY_REFRESH_RATE_LEGACY, this.refreshRateHz));

		if (null == getFilter() && (this.alpha != (byte) 0xff)) {
			// use the filter to simulate the translucent display
			int color = this.alpha;
			color <<= 24;
			Image filter = FrontPanel.getFrontPanel().newImage(getWidth(), getHeight(), color, true);
			// engine issue: have to erase the background again
			filter.fillRectangle(0, 0, getWidth(), getHeight(), color);
			setFilter(filter);
		}
	}

	@Override
	public void showYourself(boolean appearSwitchedOn) {
		DisplayBufferManager buffer = this.bufferManager;
		buffer.fillRectangle(0, 0, buffer.getWidth(), buffer.getHeight(), convertDisplayColorToARGBColor(
				convertARGBColorToDisplayColor(appearSwitchedOn ? ~getInitialColor() : getInitialColor())));
		buffer.flush();
	}

	/**
	 * Called by the parser after filling all the fields defined in the xml. Used to complete the display
	 * initialization.
	 */
	@Override
	public void start() {
		super.start();

		// load optional extension (may load an extension class which can call Device.getDevice(); so this extension
		// cannot be created in finalizeConfiguration() method)
		if (this.extensionClassName != null) {
			// load extension set in fp file
			this.extension = FrontPanel.getFrontPanel().newUserInstance(getClass(), DisplayExtension.class,
					this.extensionClassName);
		}

		// create image returned by getCurrentSkin(); this image may be smaller or higher than display buffers
		this.visibleBuffer = newWidgetImage();

		DisplayBufferPolicy bufferPolicy;
		if (this.bufferPolicyClass != null) {
			bufferPolicy = FrontPanel.getFrontPanel().newUserInstance(getClass(), DisplayBufferPolicy.class,
					this.bufferPolicyClass);
		} else {
			bufferPolicy = new SwapDoubleBufferPolicy();
		}
		BufferRefreshStrategy refreshStrategy;
		if (this.refreshStrategyClass != null) {
			refreshStrategy = FrontPanel.getFrontPanel().newUserInstance(getClass(), BufferRefreshStrategy.class,
					this.refreshStrategyClass);
		} else {
			refreshStrategy = new PredrawRefreshStrategy();
		}

		this.bufferManager = new DisplayBufferManager(bufferPolicy, refreshStrategy, this, this.displayWidth,
				this.displayHeight, this.initialColor);
		this.bufferManager.setFlushTime(this.flushTimeMs);
		this.bufferManager.setRefreshTime(this.refreshRateTimeMs);

		// turn on backlight
		setBacklight(BACKLIGHT_MAX);

		populateToolBar();
		populateStatusBar();
	}

	private void populateToolBar() {
		// Hard-coded to a specific, known UI layout.
		Container frame = getFPMainContainer();
		JToolBar toolBar = (JToolBar) frame.getComponent(TOOL_BAR_POSITION);

		ImageIcon screenshotIcon = getIcon("screenshot.png"); //$NON-NLS-1$
		this.screenshotAction = new AbstractAction(null, screenshotIcon) {
			private static final long serialVersionUID = 1L;

			@Override
			public void actionPerformed(ActionEvent e) {
				createScreenshot();
			}
		};
		this.screenshotAction.putValue(Action.SHORT_DESCRIPTION, "Screenshot"); //$NON-NLS-1$
		toolBar.add(this.screenshotAction);
		this.outputDir = Paths.get(System.getProperty("application.output.dir"), "screenshots").toFile();
		this.outputDir.mkdirs();
		toolBar.addSeparator();
	}

	private void populateStatusBar() {
		// Hard-coded to a specific, known UI layout.
		Container frame = getFPMainContainer();
		JPanel statusBar = (JPanel) frame.getComponent(STATUS_BAR_POSITION);

		this.coordinatesLabel = new JLabel(coordinatesToString(0, 0));
		this.coordinatesLabel.setEnabled(false);
		JSeparator coordinatesLabelSeparator = new JSeparator(SwingConstants.VERTICAL);
		Dimension coordinatesLabelPreferredSize = coordinatesLabelSeparator.getPreferredSize();
		coordinatesLabelPreferredSize.height = this.coordinatesLabel.getPreferredSize().height;
		coordinatesLabelSeparator.setPreferredSize(coordinatesLabelPreferredSize);
		JPanel statusBarLeft = (JPanel) statusBar.getComponent(STATUS_BAR_LEFT_POSITION);
		statusBarLeft.add(coordinatesLabelSeparator, 0);
		statusBarLeft.add(this.coordinatesLabel, 0);

		this.screenshotNotificationLabel = new JLabel("<html>A new screenshot is <u>available<u></html>");
		this.screenshotNotificationLabel.setForeground(Color.BLUE);
		this.screenshotNotificationLabel.addMouseListener(new MouseAdapter() {
			@Override
			public void mouseClicked(MouseEvent e) {
				try {
					Desktop.getDesktop().open(Display.this.outputDir);
				} catch (IOException e1) {
					// Ignore.
				}
			}
		});
		this.screenshotNotificationLabel.setVisible(false);
		JPanel statusBarCenter = (JPanel) statusBar.getComponent(STATUS_BAR_CENTER_POSITION);
		statusBarCenter.add(this.screenshotNotificationLabel);
	}

	private Container getFPMainContainer() {
		// Hard-coded to a specific, known UI layout.
		FrontPanel frontPanel = FrontPanel.getFrontPanel();
		return ((JComponent) frontPanel.getDeviceWidget()).getParent().getParent();
	}

	private ImageIcon getIcon(String name) {
		return new ImageIcon(getClass().getClassLoader().getResource(IMAGES_PREFIX + name));
	}

	private void createScreenshot() {
		SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd_HH-mm-ss");
		Date now = new Date();
		String date = dateFormat.format(now);
		File outputFile = new File(this.outputDir, "fp_screenshot_" + date + ".png");
		try {
			ImageIO.write((BufferedImage) this.visibleBuffer.getRAWImage(), "png", outputFile);
		} catch (IOException e) {
			throw new IllegalStateException("Unable to create image " + outputFile, e);
		}

		this.screenshotNotificationCounter++;
		this.screenshotNotificationLabel.setVisible(true);
		new Thread() {
			@Override
			public void run() {
				try {
					sleep(SCREENSHOT_NOTIFICATION_DURATION);
				} catch (InterruptedException e) {
					// Ignore.
				}
				if (--Display.this.screenshotNotificationCounter == 0) {
					Display.this.screenshotNotificationLabel.setVisible(false);
				}
			}
		}.start();
	}

	private void updateCoordinatesLabel(int x, int y) {
		this.coordinatesLabel.setText(coordinatesToString(x, y));
		this.coordinatesLabel.repaint();
	}

	private String coordinatesToString(int x, int y) {
		return MessageFormat.format(DISPLAY_COORDINATES_PATTERN, Integer.valueOf(x), Integer.valueOf(y));
	}

	@Override
	public void dispose() {
		this.bufferManager.dispose();
		this.bufferManager = null;
		this.visibleBuffer = null;
		super.dispose();
	}

	@Override
	public Image getDisplayedImage() {
		return getCurrentSkin();
	}

	/**
	 * Gets the current displayed skin.
	 *
	 * @return the current displayed skin.
	 */
	@Override
	public synchronized Image getCurrentSkin() {
		// draw frame buffer data on visible buffer (scale up/down if necessary)
		Image resultImage = this.visibleBuffer;
		resultImage.drawImage(this.bufferManager.getDisplayImage());

		// apply contrast and backlight
		drawContrast(resultImage);
		drawBacklight(resultImage);

		// crop displayed area according mask
		resultImage.crop(getFilter());

		return resultImage;
	}

	/**
	 * Gets the initial color of the display.
	 *
	 * @return the initial color.
	 */
	public int getInitialColor() {
		return this.initialColor;
	}

	/**
	 * Called by MicroUI graphical engine when MicroUI.start() is called by the MicroEJ application.
	 */
	@Override
	public Image initialize() {
		// nothing else to initialize, just have to return the back buffer
		return this.bufferManager;
	}

	@Override
	public boolean hasBacklight() {
		return this.hasBacklight;
	}

	/**
	 * Returns the backlight of the display. <code>backlight</code> value range is
	 * {@link #BACKLIGHT_MIN}-{@link #BACKLIGHT_MAX}. If the display do not manage backlight (
	 * <code>hasBacklight() == false</code>), returns {@link #BACKLIGHT_MIN}.
	 *
	 * @return the backlight of the display.
	 */
	@Override
	public int getBacklight() {
		return hasBacklight() ? this.backlight : BACKLIGHT_MIN;
	}

	/**
	 * Sets the backlight of the display.
	 *
	 * @param backlight
	 *            the new value of the backlight.
	 */
	@Override
	public void setBacklight(int backlight) {
		if (hasBacklight()) {
			if (this.backlight < BACKLIGHT_MIN) {
				this.backlight = BACKLIGHT_MIN;
			} else if (this.backlight > BACKLIGHT_MAX) {
				this.backlight = BACKLIGHT_MAX;
			} else {
				this.backlight = backlight;
			}
			repaint();
		}
	}

	@Override
	public boolean isColor() {
		return this.extension != null ? this.extension.isColor(this) : LLUIDisplayImpl.super.isColor();
	}

	@Override
	public int getNumberOfColors() {
		return this.extension != null ? this.extension.getNumberOfColors(this)
				: LLUIDisplayImpl.super.getNumberOfColors();
	}

	@Override
	public int convertARGBColorToDisplayColor(int argbColor) {
		return this.extension != null ? this.extension.convertARGBColorToDisplayColor(this, argbColor)
				: LLUIDisplayImpl.super.convertARGBColorToDisplayColor(argbColor);
	}

	@Override
	public int convertDisplayColorToARGBColor(int displayColor) {
		return this.extension != null ? this.extension.convertDisplayColorToARGBColor(this, displayColor)
				: LLUIDisplayImpl.super.convertDisplayColorToARGBColor(displayColor);
	}

	@Override
	public boolean prepareBlendingOfIndexedColors(AtomicInteger foreground, AtomicInteger background) {
		return this.extension != null ? this.extension.prepareBlendingOfIndexedColors(this, foreground, background)
				: LLUIDisplayImpl.super.prepareBlendingOfIndexedColors(foreground, background);
	}

	@Override
	public Image getCurrentBackBuffer() {
		return this.bufferManager.getCurrentBackBuffer();
	}

	@Override
	public Image getCurrentDrawingBuffer() {
		return this.bufferManager.getCurrentBackBuffer();
	}

	@Override
	public boolean isDoubleBuffered() {
		return this.bufferManager.isDoubleBuffered();
	}

	@Override
	public void flush() {
		this.bufferManager.flush();
	}

	@Override
	public void waitFlush() {
		this.bufferManager.waitFlush();
	}

	@Override
	public void newDrawingRegion(int x1, int y1, int x2, int y2, boolean drawingNow) {
		this.bufferManager.newDrawingRegion(x1, y1, x2, y2, drawingNow);
	}

	@Override
	public void repaint() {
		if (this.alpha != (byte) 0xff) {
			// parent must repaint the display background
			getParent().repaint(getX(), getY(), getWidth(), getHeight());
		} else {
			// all display should be repaint because the realSize and the size
			// can be different
			super.repaint();
		}
	}

	/**
	 * Creates a new image with the same size than display (not widget size).
	 *
	 * @return an image as big as display.
	 */
	protected Image newDisplayImage() {
		return FrontPanel.getFrontPanel().newImage(this.displayWidth, this.displayHeight, this.initialColor, false);
	}

	/**
	 * Creates a new image with the same size than widget (not display size).
	 *
	 * @return an image as big as widget.
	 */
	protected Image newWidgetImage() {
		return FrontPanel.getFrontPanel().newImage(getWidth(), getHeight(), this.initialColor, true);
	}

	/**
	 * Transforms the skin data applying an algorithm to simulate the display contrast.
	 * <p>
	 * The default implementation performs nothing.
	 *
	 * @param imageSkin
	 *            the image where applying the algorithm.
	 */
	protected void drawContrast(Image imageSkin) {
		// not implemented
	}

	/**
	 * Transforms the skin data applying an algorithm to simulate the display backlight.
	 *
	 * @param imageSkin
	 *            the image where applying the algorithm.
	 */
	protected void drawBacklight(Image imageSkin) {
		if (hasBacklight()) {
			// get a transparency level according backlight
			int shadow = ((BACKLIGHT_MAX - getBacklight()) * 0xff / BACKLIGHT_MAX);
			imageSkin.fillTransparentRectangle(0, 0, imageSkin.getWidth(), imageSkin.getHeight(), shadow << 24);
		}
	}

	private int getValue(String property, String legacyProperty, int defaultValue) {
		return Integer.getInteger(property, Integer.getInteger(legacyProperty, defaultValue)).intValue();
	}

	@Override
	public void mouseDragged(int x, int y) {
		mouseMoved(x, y);
	}

	@Override
	public void mouseMoved(int x, int y) {
		this.coordinatesLabel.setEnabled(true);
		updateCoordinatesLabel(x, y);
	}

	@Override
	public void mouseExited(int x, int y) {
		this.coordinatesLabel.setEnabled(false);
	}

}
