/*
 * Copyright 2015-2021 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.observable;

import ej.basictool.ArrayTools;

/**
 * This class represents an observable object, or "data" in the model-view paradigm. It can be subclassed to represent
 * an object that the application wants to have observed.
 * <p>
 * An observable object can have several observers. An observer may be any object that implements interface
 * {@link Observer}. After an observable instance changes, an application calling the observable's
 * {@link #notifyObservers()} method causes all of its observers to be notified of the change by a call to their
 * {@link Observer#update()} method.
 */
public class Observable {

	private Observer[] observers;
	private boolean changed;

	/**
	 * Constructs an observable with no observers.
	 */
	public Observable() {
		this.observers = new Observer[0];
	}

	/**
	 * Adds an observer to the set of observers for this observable.
	 * <p>
	 * If the observer is already added, nothing changes.
	 * <p>
	 * The order in which notifications will be delivered to multiple observers is not specified.
	 *
	 * @param observer
	 *            an observer to add.
	 * @throws NullPointerException
	 *             if the given observer is <code>null</code>.
	 */
	@SuppressWarnings({ "null", "unused" })
	public void addObserver(Observer observer) {
		if (observer == null) {
			throw new NullPointerException();
		}
		Observer[] observers = this.observers;
		if (!ArrayTools.contains(observers, observer)) {
			this.observers = ArrayTools.add(observers, observer);
		}
	}

	/**
	 * Deletes an observer from the set of observers of this observable.
	 *
	 * @param observer
	 *            the observer to delete.
	 */
	public void deleteObserver(Observer observer) {
		this.observers = ArrayTools.remove(this.observers, observer);
	}

	/**
	 * Clears the set of observers of this observable.
	 */
	public void deleteObservers() {
		this.observers = new Observer[0];
	}

	/**
	 * Marks this observable as having been changed. The {@link #hasChanged()} method will now return <code>true</code>.
	 */
	protected void setChanged() {
		this.changed = true;
	}

	/**
	 * Indicates that this observable has no longer changed, or that it has already notified all of its observers of its
	 * most recent change, so that the {@link #hasChanged()} method will now return <code>false</code>. This method is
	 * called automatically by the {@link #notifyObservers()} method.
	 */
	protected void clearChanged() {
		this.changed = false;
	}

	/**
	 * Gets whether or not this observable has changed.
	 *
	 * @return <code>true</code> if and only if the {@link #setChanged()} method has been called more recently than the
	 *         {@link #clearChanged()} method on this observable, <code>false</code> otherwise.
	 */
	public boolean hasChanged() {
		return this.changed;
	}

	/**
	 * Returns the number of observers of this observable.
	 *
	 * @return the number of observers of this observable.
	 */
	public int countObservers() {
		return this.observers.length;
	}

	/**
	 * If this observable has changed, as indicated by the {@link #hasChanged()} method, then notify all of its
	 * observers and then call the {@link #clearChanged()} method to indicate that this observable has no longer
	 * changed.
	 * <p>
	 * Each observer has its update method called.
	 *
	 * @see Observer#update()
	 */
	public void notifyObservers() {
		if (!this.changed) {
			return;
		}
		clearChanged();
		Observer[] observers = this.observers;
		for (Observer observer : observers) {
			observer.update();
		}
	}

	/**
	 * Gets all the observers of this observable.
	 *
	 * @return an array containing all the observers of this observable.
	 */
	public Observer[] getObservers() {
		return this.observers.clone();
	}
}
