/*
 * Copyright (c) 1998, 2013, Oracle and/or its affiliates. All rights reserved.
 * Copyright (C) 2015-2016, IS2T - EDC compliance and optimizations.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package java.security.cert;

import java.io.InputStream;

/**
 * This class defines the <i>Service Provider Interface</i> (<b>SPI</b>)
 * for the {@code CertificateFactory} class.
 * All the abstract methods in this class must be implemented by each
 * cryptographic service provider who wishes to supply the implementation
 * of a certificate factory for a particular certificate type, e.g., X.509.
 *
 * <p>Certificate factories are used to generate certificate, certification path
 * ({@code CertPath}) and certificate revocation list (CRL) objects from
 * their encodings.
 *
 * <p>A certificate factory for X.509 must return certificates that are an
 * instance of {@code java.security.cert.X509Certificate}, and CRLs
 * that are an instance of {@code java.security.cert.X509CRL}.
 *
 * @author Hemma Prafullchandra
 * @author Jan Luehe
 * @author Sean Mullan
 *
 *
 * @see CertificateFactory
 * @see Certificate
 * @see X509Certificate
 *
 * @since 1.2
 */

public abstract class CertificateFactorySpi {

	/**
	 * Generates a certificate object and initializes it with
	 * the data read from the input stream {@code inStream}.
	 *
	 * <p>In order to take advantage of the specialized certificate format
	 * supported by this certificate factory,
	 * the returned certificate object can be typecast to the corresponding
	 * certificate class. For example, if this certificate
	 * factory implements X.509 certificates, the returned certificate object
	 * can be typecast to the {@code X509Certificate} class.
	 *
	 * <p>In the case of a certificate factory for X.509 certificates, the
	 * certificate provided in {@code inStream} must be DER-encoded and
	 * may be supplied in binary or printable (Base64) encoding. If the
	 * certificate is provided in Base64 encoding, it must be bounded at
	 * the beginning by -----BEGIN CERTIFICATE-----, and must be bounded at
	 * the end by -----END CERTIFICATE-----.
	 *
	 * <p>Note that if the given input stream does not support
	 * {@link java.io.InputStream#mark(int) mark} and
	 * {@link java.io.InputStream#reset() reset}, this method will
	 * consume the entire input stream. Otherwise, each call to this
	 * method consumes one certificate and the read position of the input stream
	 * is positioned to the next available byte after the inherent
	 * end-of-certificate marker. If the data in the
	 * input stream does not contain an inherent end-of-certificate marker (other
	 * than EOF) and there is trailing data after the certificate is parsed, a
	 * {@code CertificateException} is thrown.
	 *
	 * @param inStream an input stream with the certificate data.
	 *
	 * @return a certificate object initialized with the data
	 * from the input stream.
	 *
	 * @exception CertificateException on parsing errors.
	 */
	public abstract Certificate engineGenerateCertificate(InputStream inStream)
			throws CertificateException;

}
