/*
 * Java
 *
 * Copyright 2017-2018 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.microej.kf.util;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.NoSuchElementException;

import ej.kf.Feature;
import ej.kf.Kernel;
import ej.kf.Module;

/**
 * Implementation of a {@link AbstractKFList} optimized for speed.
 *
 * Access of elements through {@link HashMap} and {@link ArrayList}.
 *
 * @see KFList2
 */
public class KFList<T> extends AbstractKFList<T> {

	private final Map<Module, ArrayList<T>> featuresToLists;

	public KFList() {
		this(4);
	}

	public KFList(int initialNbContexts) {
		featuresToLists = new HashMap<>(initialNbContexts);
	}

	@Override
	public synchronized boolean contains(Object element) {
		Module owner = Kernel.getContextOwner();
		Map<Module, ArrayList<T>> featuresToLists = this.featuresToLists;
		ArrayList<T> list = featuresToLists.get(owner);
		if (list == null) {
			return false;
		} else {
			return list.contains(element);
		}
	}

	/**
	 * @throws IllegalArgumentException
	 *             if this method is called by the Kernel and the element is owned by a Feature.
	 */
	@Override
	public synchronized boolean add(T element) {
		Module owner = Kernel.getContextOwner();
		if (owner == Kernel.getInstance() && Kernel.getOwner(element) != owner) {
			throw new IllegalArgumentException(); // TODO P0095F-213
		}
		Map<Module, ArrayList<T>> featuresToLists = this.featuresToLists;
		ArrayList<T> list = featuresToLists.get(owner);
		if (list == null) {
			// this is the first time this Feature context adds an element
			list = new ArrayList<T>(2);
			Kernel.enter();
			try {
				featuresToLists.put(owner, list);
			} finally {
				Kernel.exit();
			}
		}
		return list.add(element);
	}

	@Override
	public synchronized boolean remove(Object element) {
		Module owner = Kernel.getContextOwner();
		if (owner == Kernel.getInstance() && Kernel.getOwner(element) != owner) {
			throw new IllegalArgumentException(); // TODO P0095F-213
		}
		Map<Module, ArrayList<T>> featuresToLists = this.featuresToLists;
		ArrayList<T> list;
		synchronized (featuresToLists) {
			// synchronized access to the map a
			list = featuresToLists.get(owner);
			if (list == null) {
				return false;
			}
		}

		boolean result = list.remove(element);

		if (list.isEmpty()) {
			// atomic modification of the map in Kernel mode
			Kernel.enter();
			try {
				synchronized (featuresToLists) {
					featuresToLists.remove(owner);
				}
			} finally {
				Kernel.exit();
			}
		}

		return result;
	}

	@Override
	public Iterator<T> iterator() {
		Map<Module, ArrayList<T>> featuresToLists = this.featuresToLists;
		if (Kernel.isInKernelMode()) {
			Collection<ArrayList<T>> subArrayValues = featuresToLists.values();
			@SuppressWarnings("unchecked")
			ArrayList<T>[] values = new ArrayList[subArrayValues.size()];
			subArrayValues.toArray(values);
			return new CompositeIterator(values);
		} else {
			Module owner = Kernel.getContextOwner();
			ArrayList<T> list;
			synchronized (featuresToLists) {
				list = featuresToLists.get(owner);
			}
			if (list == null) {
				return new ArrayList<T>(0).iterator(); // empty iterator
			} else {
				return list.iterator();
			}
		}
	}

	class CompositeIterator implements Iterator<T> {

		private final ArrayList<T>[] subArrays;
		private Iterator<T> currentIterator; // null when not computed or when end of the composite iterator is reached
		private int nextIteratorIndex; // 0 based. Position of the next iterator.

		public CompositeIterator(ArrayList<T>[] subArrays) {
			this.subArrays = subArrays;
			this.nextIteratorIndex = 0;
		}

		@Override
		public boolean hasNext() {
			// Implementation Note: when this method returns true, currentIterator is not null for sure
			int nbIterators = subArrays.length;
			while (nextIteratorIndex < nbIterators) {
				if (currentIterator == null) {
					currentIterator = subArrays[nextIteratorIndex].iterator();
				}
				if (currentIterator.hasNext()) {
					return true;
				}
				currentIterator = null;
				++nextIteratorIndex;
			}
			return false;
		}

		@Override
		public T next() {
			if (hasNext()) {
				return currentIterator.next();
			}
			throw new NoSuchElementException();
		}

		@Override
		public void remove() {
			if (currentIterator == null) {
				throw new IllegalStateException();
			}
			currentIterator.remove();
		}

	}

	@Override
	public synchronized void clear() {
		if (Kernel.isInKernelMode()) {
			featuresToLists.clear();
		} else {
			Module owner = Kernel.getContextOwner();
			ArrayList<T> list = featuresToLists.get(owner);
			if (list != null) {
				list.clear();
			}
		}
	}

	@Override
	public synchronized boolean isEmpty() {
		if (Kernel.isInKernelMode()) {
			return featuresToLists.isEmpty();
		} else {
			Module owner = Kernel.getContextOwner();
			ArrayList<T> list = featuresToLists.get(owner);
			if (list == null) {
				return true;
			} else {
				return list.isEmpty();
			}
		}
	}

	@Override
	public synchronized int size() {
		int size;
		if (Kernel.isInKernelMode()) {
			size = 0;
			for (ArrayList<T> subArrayValues : featuresToLists.values()) {
				size += subArrayValues.size();
			}
		} else {
			Module owner = Kernel.getContextOwner();
			ArrayList<T> list = featuresToLists.get(owner);
			if (list == null) {
				size = 0;
			} else {
				size = list.size();
			}
		}
		return size;
	}

	@Override
	protected void toArray(ArrayList<T> allList) {
		if (Kernel.isInKernelMode()) {
			for (ArrayList<T> subArrayValues : featuresToLists.values()) {
				allList.addAll(subArrayValues);
			}
		} else {
			Module owner = Kernel.getContextOwner();
			ArrayList<T> list = featuresToLists.get(owner);
			if (list != null) {
				allList.addAll(list);
			}
		}
	}

	@Override
	protected void removeAllElementsOwnedBy(Feature feature) {
		featuresToLists.remove(feature);
	}
}
