/*
 * Java
 *
 * Copyright 2011-2019 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.mwt;

import ej.annotation.NonNull;
import ej.annotation.Nullable;
import ej.microui.MicroUI;
import ej.microui.display.Colors;
import ej.microui.display.Font;
import ej.microui.display.GraphicsContext;
import ej.microui.event.Event;
import ej.microui.event.generator.Command;
import ej.microui.util.EventHandler;

/*package*/class RenderableHelper {

	private static Font DefaultFont;

	private RenderableHelper() {
	}

	/**
	 * Paints the renderable.
	 * <ol>
	 * <li>Check that it is shown on a desktop.</li>
	 * <li>Check that it can be painted respecting the "view from" (see {@link Desktop}).</li>
	 * <li>Check that the clip is not empty.</li>
	 * <li>Check the renderer is not null</li>
	 * <li>Call {@link Renderer#render(GraphicsContext, Renderable)}</li>
	 * </ol>
	 *
	 * @param g
	 *            the graphics context.
	 * @param desktop
	 *            the desktop of the renderable.
	 * @param renderable
	 *            the renderable to paint.
	 */
	/* package */static void paintRenderable(@NonNull GraphicsContext g, @Nullable Desktop desktop,
			@NonNull Renderable renderable) {
		if (desktop == null) {
			return;
		}
		// check repaint
		if (!desktop.checkPaintFrom(renderable)) {
			return;
		}
		if (g.getClipWidth() == 0 || g.getClipHeight() == 0) {
			return; // nothing to paint
		}
		// call the render
		reset(g);
		renderable.render(g);
	}

	private static void reset(@NonNull GraphicsContext g) {
		// spec: current color is black
		g.setColor(Colors.BLACK);
		g.removeBackgroundColor();
		// and the font is the default font
		Font defaultFont = getDefaultFont();
		if (defaultFont != null) {
			// when there is no font, do not throw the NPE here (not user code)
			defaultFont.resetRatios();
		}
		g.setFont(defaultFont);
		// and stroke style is SOLID
		g.setStrokeStyle(GraphicsContext.SOLID);
		g.setEllipsis(false);
	}

	@NonNull
	private static Font getDefaultFont() {
		if (DefaultFont == null) {
			DefaultFont = Font.getDefaultFont();
		}
		return DefaultFont;
	}

	/* package */static int getDirection(int event) {
		if (Event.getType(event) == Event.COMMAND) {
			int data = Event.getData(event);
			switch (data) {
			case Command.UP:
				return MWT.UP;
			case Command.LEFT:
				return MWT.LEFT;
			case Command.DOWN:
				return MWT.DOWN;
			case Command.RIGHT:
				return MWT.RIGHT;
			}
		}
		return MWT.NONE;
	}

	/* package */static boolean handleEvent(int event, @Nullable EventHandler eventHandler) {
		if (eventHandler != null) {
			try {
				return eventHandler.handleEvent(event);
			} catch (Throwable e) {
				// TODO use logger
				// exception occurred in a renderer method
				MicroUI.MicroUI.errorLog(e);
			}
		}
		return false;
	}

}
