/*
 * Java
 *
 * Copyright 2024-2025 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.kf.util.security;

import java.io.IOException;
import java.io.InputStream;
import java.security.Permission;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Logger;

import ej.annotation.Nullable;
import ej.basictool.map.PackedMap;
import ej.kf.Feature;
import ej.kf.Feature.State;
import ej.kf.FeatureStateListener;
import ej.kf.Kernel;

/**
 * This class contains the security logic to apply when working with feature policy files.
 */
public class KernelSecurityPolicyManager extends SecurityManager implements FeatureStateListener {

	private static final Logger LOGGER = Logger.getLogger(KernelSecurityPolicyManager.class.getName());
	private final PackedMap<Feature, List<FeaturePolicyPermission>> featurePermissionsMap;
	private final SecurityPolicyResourceLoader securityPolicyResourceLoader;

	/**
	 * Instantiates a new kernel security policy manager.
	 *
	 * @param securityPolicyResourceLoader
	 *            the security policy resource loader implementation.
	 */
	public KernelSecurityPolicyManager(SecurityPolicyResourceLoader securityPolicyResourceLoader) {
		this.featurePermissionsMap = new PackedMap<>();
		this.securityPolicyResourceLoader = securityPolicyResourceLoader;
		handleFeatureStateUpdates();
	}

	private void handleFeatureStateUpdates() {
		Kernel.addFeatureStateListener(this);
	}

	@Override
	public void checkPermission(Permission perm) {

		if (!Kernel.isInKernelMode()) {
			Feature f = (Feature) Kernel.getContextOwner();
			Kernel.enter();
			List<FeaturePolicyPermission> featurePermissionList = featurePermissionsMap.get(f);

			if (featurePermissionList != null) {
				checkPermission(perm, f);
			} else {
				// the feature has not registered any permissions
				throw new SecurityException();
			}
		}

		// else: Kernel has all the rights: no checks
	}

	private void checkPermission(Permission perm, Feature feature) {

		LOGGER.fine("Checking permission for: " + perm.getClass().getName() + " name: " + perm.getName() + " actions: "
				+ perm.getActions());

		List<FeaturePolicyPermission> permissionList = featurePermissionsMap.get(feature);

		boolean isAllowed = false;

		for (FeaturePolicyPermission policyPermission : permissionList) {

			// check if the permission class name fits the registered permission

			if (policyPermission.getPermissionClassName().equals(SecurityPolicyResourceLoader.ALL_IDENTIFIER)
					|| policyPermission.getPermissionClassName().equals(perm.getClass().getName())) {

				// if permission class name is here but map is empty means allow everything
				PackedMap<String, List<String>> nameActionsMap = policyPermission.getNameActionsMap();

				if (nameActionsMap.isEmpty()) {
					isAllowed = true;
					break; // NOSONAR Improve code efficiency
				}

				// loop through map for the name
				String bestNameMatch = findBestNameMatch(nameActionsMap, perm);

				// couldn't have any equal or wildcard implied match
				if (bestNameMatch.isEmpty()) {
					break; // NOSONAR Improve code efficiency
				}

				List<String> actionList = policyPermission.getNameActionsMap().get(bestNameMatch);

				if (actionList != null) {
					isAllowed = isActionPresent(actionList, perm);

				}

				// if isAllowed has been updated, further check are useless
				if (isAllowed) {
					break;
				}

			}
		}

		// if we reach here and allowed is still at false, it didn't fit in any of the
		// previous cases, throw exception
		if (!isAllowed) {
			throw new SecurityException(feature.getName() + " is not allowed to access " + perm.getClass().getName()
					+ " (name: " + perm.getName() + " - actions: " + perm.getActions() + ")");
		} else {
			LOGGER.fine("Permission granted for: " + feature.getName() + " for permission: " + perm.getClass().getName()
					+ " name: " + perm.getName() + " actions: " + perm.getActions());
		}
	}

	/**
	 * Split string array.
	 *
	 * @param input
	 *            the input
	 * @param splitter
	 *            the splitter
	 * @return the list
	 */
	private List<String> splitStringArray(String input, char splitter) {

		List<String> result = new ArrayList<>();

		// Perform the split operation
		int startIndex = 0;
		for (int i = 0; i < input.length(); i++) {
			if (input.charAt(i) == splitter) {
				// Extract substring from startIndex to i (excluding i)
				result.add(input.substring(startIndex, i));
				// Update startIndex to next character position
				startIndex = i + 1;
			}
		}

		// Add the remaining substring after the last delimiter (or the whole string if
		// no delimiter found)
		result.add(input.substring(startIndex));

		return result;
	}

	/**
	 * This method allows to load the feature's permissions. This method is already called by the
	 * {@link KernelSecurityPolicyManager} class when using standard LocalDeploy. This method only needs to be called
	 * when the Kernel uses XIP (Execution In Place) system to load features permissions before starting them. See
	 * <a href= "https://docs.microej.com/en/latest/VEEPortingGuide/multiSandbox.html">the documentation</a> for more
	 * informations.
	 *
	 * @param feature
	 *            the feature to load permissions from.
	 */
	public void addToPermissionMap(Feature feature) {
		String resourceFileName = System.getProperty("feature.policy.name", "/feature.policy.json");
		try (InputStream inputStream = feature
				.getResourceAsStream(resourceFileName.startsWith("/") ? resourceFileName : "/" + resourceFileName)) {
			List<FeaturePolicyPermission> permissionList = securityPolicyResourceLoader
					.loadFeaturePermissions(inputStream);
			featurePermissionsMap.put(feature,
					permissionList == null ? new ArrayList<FeaturePolicyPermission>() : permissionList);
		} catch (IOException e) {
			LOGGER.severe(e.getMessage());
		}
	}

	private String findBestNameMatch(PackedMap<String, List<String>> nameActionsMap, Permission perm) {
		String bestNameMatch = "";

		for (String name : nameActionsMap.keySet()) {

			// match equals name or null (perfect match)
			if ((perm.getName() == null && name.equals(SecurityPolicyResourceLoader.NULL_IDENTIFIER)
					|| (perm.getName() != null && name.equals(perm.getName())))) {
				bestNameMatch = name;
				break;

				// name ends with wildcard, and contains the perm name, finally the path is more
				// precise, update bestNameMatch
			} else if (name.endsWith(SecurityPolicyResourceLoader.ALL_IDENTIFIER)
					&& (perm.getName() == null || perm.getName().contains(name.substring(0, name.length() - 1)))
					&& name.length() > bestNameMatch.length()) {
				bestNameMatch = name;
			}
		}

		return bestNameMatch;
	}

	private boolean isActionPresent(List<String> actionList, Permission perm) {
		boolean isActionPresent = false;

		if (actionList.isEmpty() || actionList.contains(SecurityPolicyResourceLoader.ALL_IDENTIFIER)
				|| (perm.getActions() == null && actionList.contains(SecurityPolicyResourceLoader.NULL_IDENTIFIER))
				|| (perm.getActions() != null && actionList.containsAll(splitStringArray(perm.getActions(), ',')))) {
			isActionPresent = true;
		}

		return isActionPresent;
	}

	@Override
	public void stateChanged(Feature feature, @Nullable State previousState) {

		if (feature.getState().equals(State.INSTALLED) && previousState == null) {
			addToPermissionMap(feature);
		}

		else if (feature.getState().equals(State.UNINSTALLED)
				&& (previousState != null && previousState.equals(State.STOPPED))) {
			featurePermissionsMap.remove(feature);
		}

	}

}
