/*
 * Java
 *
 *	Copyright 2025 MicroEJ Corp. All rights reserved.
 *	Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.kf.util.module;

import com.microej.kf.util.control.fs.FileSystemResourcesController;
import com.microej.kf.util.control.net.NetResourcesController;
import com.microej.kf.util.policy.FeaturePolicyEnforcer;
import ej.kf.Feature;
import ej.kf.Feature.State;
import ej.kf.Kernel;
import ej.kf.Module;
import org.json.me.JSONException;

import java.io.IOException;

/**
 * Application module based on KF.
 */
public class ApplicationModule implements SandboxedModule {

	private final Feature feature;
	/** The file system usage controller of this application. */
	private final FileSystemResourcesController fsController;
	/** The network resource usage controller of this application. */
	private final NetResourcesController netController;
	/** The policy enforcer of this application. */
	private final FeaturePolicyEnforcer appPolicyEnforcer;

	protected ApplicationModule(Feature feature) {
		this.feature = feature;
		this.fsController = new FileSystemResourcesController((Module) feature);
		this.netController = new NetResourcesController();
		this.appPolicyEnforcer = new FeaturePolicyEnforcer();
	}

	@Override
	public FileSystemResourcesController getFileSystemResourceController() {
		return fsController;
	}

	@Override
	public NetResourcesController getNetworkResourceController() {
		return netController;
	}

	@Override
	public String getIdentifier() {
		return this.feature.getName();
	}

	@Override
	public Module getModule() {
		return this.feature;
	}

	@Override
	public State getState() {
		Kernel.enter();
		switch (this.feature.getState()) {
		case UNINSTALLED:
			return State.UNINSTALLED;
		case STARTED:
			return State.STARTED;
		case STOPPED:
		case INSTALLED:
			return State.INSTALLED;
		default:
			throw new IllegalStateException();
		}
	}

	public void applyPolicy() throws JSONException, IOException {
		this.appPolicyEnforcer.applyPolicy(this.feature);
	}

	public void applyThreadPriority() {
		this.appPolicyEnforcer.applyThreadPriority(this.feature);
	}
}
