/*******************************************************************************
 * Copyright (c) 2009, 2016 IBM Corp.
 * Copyright 2019-2021 MicroEJ Corp. This file has been modified by MicroEJ Corp.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * and Eclipse Distribution License v1.0 which accompany this distribution.
 *
 * The Eclipse Public License is available at
 *    http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 *   http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *    Dave Locke - initial API and implementation and/or initial documentation
 *    Ian Craggs - MQTT 3.1.1 support
 *    James Sutton - Automatic Reconnect & Offline Buffering
 *    MicroEJ Corp. - MicroPaho implementation and optimizations on MicroEJ
 */
package org.eclipse.paho.client.mqttv3;

import ej.annotation.NonNull;

/**
 * Thrown if an error occurs communicating with the server.
 */
public class MqttException extends RuntimeException {

	/**
	 * Client encountered an exception. Use the {@link #getCause()} method to get the underlying reason.
	 */
	public static final int REASON_CODE_CLIENT_EXCEPTION = 0x00;

	/** Error from subscribe - returned from the server. */
	public static final int REASON_CODE_SUBSCRIBE_FAILED = 0x80;

	/**
	 * Client timed out while waiting for a response from the server. The server is no longer responding to keep-alive
	 * messages.
	 */
	public static final int REASON_CODE_CLIENT_TIMEOUT = 32000;

	/**
	 * The client is already connected.
	 */
	public static final int REASON_CODE_CLIENT_CONNECTED = 32100;

	/**
	 * The client is already disconnected.
	 */
	public static final short REASON_CODE_CLIENT_ALREADY_DISCONNECTED = 32101;

	/** Unable to connect to server. */
	public static final int REASON_CODE_SERVER_CONNECT_ERROR = 32103;

	/**
	 * The client is not connected to the server. The {@link MqttClient#connect()} or
	 * {@link MqttClient#connect(MqttConnectOptions)} method must be called first. It is also possible that the
	 * connection was lost - see {@link MqttClient#setCallback(MqttCallback)} for a way to track lost connections.
	 */
	public static final int REASON_CODE_CLIENT_NOT_CONNECTED = 32104;

	/**
	 * Protocol error: the message was not recognized as a valid MQTT packet. Possible reasons for this include
	 * connecting to a non-MQTT server, or connecting to an SSL server port when the client isn't using SSL.
	 */
	public static final int REASON_CODE_INVALID_MESSAGE = 32108;

	/**
	 * The client has been unexpectedly disconnected from the server. The {@link #getCause() cause} will provide more
	 * details.
	 */
	public static final int REASON_CODE_CONNECTION_LOST = 32109;

	/**
	 * The client is closed - no operations are permitted on the client in this state. New up a new client to continue.
	 */
	public static final int REASON_CODE_CLIENT_CLOSED = 32111;

	/**
	 * Server inconsistency (message length integer overflow).
	 */
	public static final int REASON_CODE_SERVER_MESSAGE_LENGTH_OVERFLOW = 44000;
	/**
	 * Server inconsistency (clean session not accepted).
	 */
	public static final int REASON_CODE_SERVER_EXPECTED_CLEAN_SESSION = 440001;

	/**
	 * Server inconsistency (received message acknowledgment to unexpected action).
	 */
	public static final int REASON_CODE_SERVER_UNEXPECTED_ACK_MESSAGE = 44002;

	/**
	 * Server inconsistency (received message acknowledgment to unexpected packet identifier).
	 */
	public static final int REASON_CODE_SERVER_UNEXPECTED_ACK_PACKET_IDENTIFIER = 44003;

	/**
	 * Server inconsistency (received unexpected ping response packet).
	 */
	public static final int REASON_CODE_SERVER_UNEXPECTED_PINGRESP_MESSAGE = 44004;

	private final int reasonCode;

	/**
	 * Constructs a new <code>MqttException</code> with the specified code as the underlying reason.
	 *
	 * @param reasonCode
	 *            the reason code for the exception.
	 */
	public MqttException(int reasonCode) {
		super();
		this.reasonCode = reasonCode;
	}

	/**
	 * Constructs a new <code>MqttException</code> with the specified <code>Throwable</code> as the underlying reason.
	 *
	 * @param cause
	 *            the underlying cause of the exception.
	 */
	public MqttException(Throwable cause) {
		super(cause);
		this.reasonCode = REASON_CODE_CLIENT_EXCEPTION;
	}

	/**
	 * Constructs a new <code>MqttException</code> with the specified <code>Throwable</code> as the underlying reason.
	 *
	 * @param reason
	 *            the reason code for the exception.
	 * @param cause
	 *            the underlying cause of the exception.
	 */
	public MqttException(int reason, Throwable cause) {
		super(cause);
		this.reasonCode = reason;
	}

	/**
	 * Returns the reason code for this exception.
	 *
	 * @return the code representing the reason for this exception.
	 */
	public int getReasonCode() {
		return this.reasonCode;
	}

	/**
	 * Returns the detail message for this exception.
	 *
	 * @return the detail message, which may be <code>null</code>.
	 */
	@Override
	@NonNull
	public String getMessage() {
		return MqttClient.MQTT + this.reasonCode;
	}

}
