/*
 * Copyright 2015-2020 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.mwt.style.outline.border;

import ej.annotation.Nullable;
import ej.bon.XMath;
import ej.microui.display.GraphicsContext;
import ej.microui.display.Painter;
import ej.mwt.style.outline.Outline;
import ej.mwt.util.Outlineable;
import ej.mwt.util.Size;

/**
 * Represents and draw a rectangular border with different thicknesses and colors.
 * <p>
 * Each edge color can be customized.
 * <p>
 * The thicknesses are stored as a <code>char</code> for heap optimization and therefore cannot exceed
 * <code>65535</code>.
 */
public class FlexibleRectangularMulticoloredBorder implements Outline {

	private final char top;
	private final char bottom;
	private final char left;
	private final char right;
	private final int colorTop;
	private final int colorBottom;
	private final int colorLeft;
	private final int colorRight;

	/**
	 * Creates a flexible rectangular border specifying its edges thickness and color.
	 * <p>
	 * The given thickness values are clamped between <code>0</code> and <code>Character.MAX_VALUE</code>.
	 *
	 * @param top
	 *            the top thickness to set.
	 * @param colorTop
	 *            the top color to set.
	 * @param right
	 *            the right thickness to set.
	 * @param colorRight
	 *            the right color to set.
	 * @param bottom
	 *            the bottom thickness to set.
	 * @param colorBottom
	 *            the bottom color to set.
	 * @param left
	 *            the left thickness to set.
	 * @param colorLeft
	 *            the left color to set.
	 */
	public FlexibleRectangularMulticoloredBorder(int top, int colorTop, int right, int colorRight, int bottom,
			int colorBottom, int left, int colorLeft) {
		this.top = (char) XMath.limit(top, 0, Character.MAX_VALUE);
		this.right = (char) XMath.limit(right, 0, Character.MAX_VALUE);
		this.bottom = (char) XMath.limit(bottom, 0, Character.MAX_VALUE);
		this.left = (char) XMath.limit(left, 0, Character.MAX_VALUE);
		this.colorTop = colorTop;
		this.colorRight = colorRight;
		this.colorBottom = colorBottom;
		this.colorLeft = colorLeft;
	}

	/**
	 * Gets the top color.
	 *
	 * @return the top color.
	 */
	public int getColorTop() {
		return this.colorTop;
	}

	/**
	 * Gets the bottom color.
	 *
	 * @return the bottom color.
	 */
	public int getColorBottom() {
		return this.colorBottom;
	}

	/**
	 * Gets the left color.
	 *
	 * @return the left color.
	 */
	public int getColorLeft() {
		return this.colorLeft;
	}

	/**
	 * Gets the right color.
	 *
	 * @return the right color.
	 */
	public int getColorRight() {
		return this.colorRight;
	}

	/**
	 * Gets the top thickness.
	 *
	 * @return the top thickness.
	 */
	public int getTop() {
		return this.top;
	}

	/**
	 * Gets the bottom thickness.
	 *
	 * @return the bottom thickness.
	 */
	public int getBottom() {
		return this.bottom;
	}

	/**
	 * Gets the left thickness.
	 *
	 * @return the left thickness.
	 */
	public int getLeft() {
		return this.left;
	}

	/**
	 * Gets the right thickness.
	 *
	 * @return the right thickness.
	 */
	public int getRight() {
		return this.right;
	}

	@Override
	public void apply(Outlineable outlineable) {
		outlineable.removeOutline(this.left, this.top, this.right, this.bottom);
	}

	@Override
	public void apply(GraphicsContext g, Size size) {
		int width = size.getWidth();
		int height = size.getHeight();

		// TODO manage corners
		g.setColor(this.colorTop);
		int topThickness = getTop();
		Painter.fillRectangle(g, 0, 0, width, topThickness);

		g.setColor(this.colorBottom);
		int bottomThickness = getBottom();
		Painter.fillRectangle(g, 0, height - bottomThickness, width, bottomThickness);

		g.setColor(this.colorLeft);
		int leftThickness = getLeft();
		Painter.fillRectangle(g, 0, 0, leftThickness, height);

		g.setColor(this.colorRight);
		int rightThickness = getRight();
		Painter.fillRectangle(g, width - rightThickness, 0, rightThickness, height);

		size.removeOutline(leftThickness, topThickness, rightThickness, bottomThickness);

		g.translate(leftThickness, topThickness);
		g.intersectClip(0, 0, size.getWidth(), size.getHeight());
	}

	@Override
	public boolean equals(@Nullable Object obj) {
		if (obj instanceof FlexibleRectangularMulticoloredBorder) {
			FlexibleRectangularMulticoloredBorder border = (FlexibleRectangularMulticoloredBorder) obj;
			return this.top == border.top && this.bottom == border.bottom && this.left == border.left
					&& this.right == border.right && this.colorTop == border.colorTop
					&& this.colorBottom == border.colorBottom && this.colorLeft == border.colorLeft
					&& this.colorRight == border.colorRight;
		}
		return false;
	}

	@Override
	public int hashCode() {
		return this.top * this.bottom * this.left * this.right * this.colorTop * this.colorBottom * this.colorLeft
				* this.colorRight;
	}

}
