/*
 * Copyright 2020 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.mwt.event;

import ej.microui.event.Event;
import ej.microui.event.EventGenerator;

/**
 * This event generator is used by the {@link EventDispatcher event dispatcher} of the desktop in order to send
 * additional events to the widgets.
 */
public class DesktopEventGenerator extends EventGenerator {

	/**
	 * The DESKTOP event type.
	 */
	public static final int EVENT_TYPE = 0x04;

	private static final int ACTION_MASK = 0x000000ff;
	private static final int ACTION_SHIFT = 0;

	@Override
	public int getEventType() {
		return EVENT_TYPE;
	}

	/**
	 * Builds an event for the given action.
	 *
	 * @param action
	 *            the event action.
	 * @return the event.
	 */
	public int buildEvent(int action) {
		return Event.buildEvent(EVENT_TYPE, this, buildEventData(action));
	}

	/**
	 * Sends a desktop event for the given action to the event handler of this event generator.
	 * <p>
	 * This method is useful when other input mechanisms wish to simulate button actions.
	 *
	 * @param action
	 *            the event action.
	 */
	public void send(int action) {
		sendEvent(buildEvent(action));
	}

	/**
	 * Returns the action held by the given desktop event.
	 *
	 * @param event
	 *            the desktop event.
	 * @return the action held by the given desktop event.
	 */
	public static int getAction(int event) {
		return (event & ACTION_MASK) >>> ACTION_SHIFT;
	}

	private static int buildEventData(int action) {
		return ((action << ACTION_SHIFT) & ACTION_MASK);
	}
}
