/*
 * Java
 *
 * Copyright 2014-2019 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.websocket;

/**
 * RFC6455 defined a list a status codes (section 7.4.1) and reserved code ranges (section 7.4.2). This class provided
 * these codes and methods to validate that a given number is not a reserved value.
 * <p>
 * According to Section 5.5.1, a status code is a "2-byte unsigned integer (in network byte order)". In Java, we will
 * use 'int' (signed 4-byte integer) to store it.
 * <p>
 * Note that comments for public constants are taken from the RFC6455.
 *
 * @see ReasonForClosure
 *
 *
 *
 */
public final class CloseCodes {

	/**
	 * 1000 indicates a normal closure, meaning that the purpose for which the connection was established has been
	 * fulfilled.
	 */
	public static final int NORMAL_CLOSURE = 1000;

	/**
	 * 1001 indicates that an endpoint is "going away", such as a server going down or a browser having navigated away
	 * from a page.
	 */
	public static final int GOING_AWAY = 1001;

	/**
	 * 1002 indicates that an endpoint is terminating the connection due to a protocol error.
	 */
	public static final int PROTOCOL_ERROR = 1002;

	/**
	 * 1003 indicates that an endpoint is terminating the connection because it has received a type of data it cannot
	 * accept (e.g., an endpoint that understands only text data MAY send this if it receives a binary message).
	 */
	public static final int INVALID_INCOMING_DATA_TYPE = 1003;

	/**
	 * Reserved. The specific meaning might be defined in the future.
	 */
	public static final int RESERVED_1004 = 1004;

	/**
	 * 1005 is a reserved value and MUST NOT be set as a status code in a Close control frame by an endpoint. It is
	 * designated for use in applications expecting a status code to indicate that no status code was actually present.
	 */
	public static final int NO_STATUS_PROVIDED = 1005;

	/**
	 * 1006 is a reserved value and MUST NOT be set as a status code in a Close control frame by an endpoint. It is
	 * designated for use in applications expecting a status code to indicate that the connection was closed abnormally,
	 * e.g., without sending or receiving a Close control frame.
	 */
	public static final int CONNECTION_CLOSED_ABNORMALLY = 1006;

	/**
	 * 1007 indicates that an endpoint is terminating the connection because it has received data within a message that
	 * was not consistent with the type of the message (e.g., non-UTF-8 [RFC3629] data within a text message).
	 */
	public static final int RECEIVED_INCONSISTENT_DATA = 1007;

	/**
	 * 1008 indicates that an endpoint is terminating the connection because it has received a message that violates its
	 * policy. This is a generic status code that can be returned when there is no other more suitable status code
	 * (e.g., 1003 or 1009) or if there is a need to hide specific details about the policy.
	 */
	public static final int POLICY_VIOLATION = 1008;

	/**
	 * 1009 indicates that an endpoint is terminating the connection because it has received a message that is too big
	 * for it to process.
	 */
	public static final int MESSAGE_TO_BIG = 1009;

	/**
	 * 1010 indicates that an endpoint (client) is terminating the connection because it has expected the server to
	 * negotiate one or more extension, but the server didn't return them in the response message of the WebSocket
	 * handshake. The list of extensions that are needed SHOULD appear in the /reason/ part of the Close frame. Note
	 * that this status code is not used by the server, because it can fail the WebSocket handshake instead.
	 */
	public static final int CLIENT_EXPECTED_EXTENSIONS_FROM_SERVER = 1010;

	/**
	 * 1011 indicates that a server is terminating the connection because it encountered an unexpected condition that
	 * prevented it from fulfilling the request.
	 */
	public static final int SERVER_CANNOT_FULFILL_REQUEST = 1011;

	/**
	 * 1015 is a reserved value and MUST NOT be set as a status code in a Close control frame by an endpoint. It is
	 * designated for use in applications expecting a status code to indicate that the connection was closed due to a
	 * failure to perform a TLS handshake (e.g., the server certificate can't be verified).
	 */
	public static final int TLS_HANDSHAKE_FAILED = 1015;

	private CloseCodes() {
		// Hide utility class constructor
	}

	/**
	 * Tell if can be used in a close control frame.
	 *
	 * See sections:
	 * <ul>
	 * <li>7.4.2. Reserved Status Code Range.
	 * <li>7.4.1. Defined Status Codes
	 * </ul>
	 *
	 * @param code
	 *            the code to test
	 * @return true if 'code' is valid, 'false' otherwise.
	 */
	public static boolean canBeUsedInCloseFrame(int code) {
		// Codes in the range [0-999] are not be used
		boolean isInvalid = code >= 0 && code < NORMAL_CLOSURE;

		// "Status codes in the range 1000-2999 are reserved for definition by this protocol"
		// Only some codes in this range can be used
		isInvalid |= code >= NORMAL_CLOSURE && code <= 2999
				&& !(code <= INVALID_INCOMING_DATA_TYPE
				|| (code >= RECEIVED_INCONSISTENT_DATA && code <= SERVER_CANNOT_FULFILL_REQUEST));

		// Codes cannot be greater than 5000
		isInvalid |= code >= 5000;

		return !isInvalid;
	}

}
