/*
 * Java
 *
 * Copyright 2020-2022 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microui.display;

import java.util.ServiceLoader;

import ej.drawing.DWDrawing;
import ej.microui.display.LLUIPainter.MicroUIGraphicsContext;
import ej.microui.display.LLUIPainter.MicroUIImage;
import ej.microui.display.LLUIPainter.MicroUIImageFormat;

/**
 * This interface represents the MicroUI graphical engine (MicroUI framework). It provides some methods to map MicroUI
 * byte arrays in MicroUI Graphics Context objects, to manipulate MicroUI colors etc.
 * <p>
 * An instance of this framework is available via the {@link #Instance} field. To use the framework methods, caller must
 * synchronize its calls on this {@link #Instance} field.
 */
public interface LLUIDisplay {

	/**
	 * Unique instance of MicroUI graphical engine. It provides some methods to synchronize drawings between them, to
	 * map MicroUI byte arrays in graphical objects, to manipulate MicroUI colors etc.
	 * <p>
	 * To use the framework methods, caller must synchronize its calls on this {@link #Instance} field.
	 */
	LLUIDisplay Instance = ServiceLoader.load(LLUIDisplay.class, LLUIDisplay.class.getClassLoader()).iterator().next();

	/**
	 * Unique instance of MicroUI graphical engine's internal software drawer. It is an instance of {@link UIDrawing}
	 * and it implements all drawings in software. The rendering is exactly the same than the embedded algorithms listed
	 * in <code>ui_drawing_soft.h</code>.
	 * <p>
	 * This drawer converts the MicroUI graphics context color, respects the clip and updates the new
	 * <code>Display.flush()</code> area (dirty area) as mentioned in {@link LLUIPainter} class comment.
	 *
	 * @return an implementation of {@link UIDrawing} in software.
	 */
	UIDrawing getUIDrawerSoftware();

	/**
	 * Unique instance of MicroUI graphical engine's internal software drawer. It is an instance of {@link DWDrawing}
	 * and it implements all drawings in software. The rendering is exactly the same than the embedded algorithms listed
	 * in <code>dw_drawing_soft.h</code>.
	 * <p>
	 * This drawer converts the MicroUI graphics context color, respects the clip and updates the new
	 * <code>Display.flush()</code> area (dirty area) as mentioned in {@link LLUIPainter} class comment.
	 *
	 * @return an implementation of {@link DWDrawing} in software.
	 */
	DWDrawing getDWDrawerSoftware();

	/**
	 * Maps a MicroUI GraphicsContext byte array (retrieved by MicroEJ application calling
	 * <code>gc.getSNIContext()</code>) on a {@link MicroUIGraphicsContext}.
	 * <p>
	 * Implementation notes:
	 * <ul>
	 * <li>Same object is always returned, only internal {@link MicroUIGraphicsContext} fields are updated. By
	 * consequence this method can only be called once during a MicroUI native call. See
	 * {@link #newMicroUIGraphicsContext(byte[])} to force to create a new {@link MicroUIGraphicsContext} instead.</li>
	 * <li>The implementation assumes given byte array has been retrieved by MicroEJ application calling
	 * <code>gc.getSNIContext()</code>. If given byte array does not respect this rule, the behavior is unknown.</li>
	 * </ul>
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext representation.
	 * @return the unique {@link MicroUIGraphicsContext} object updated with the given MicroUI GraphicsContext
	 *         characteristics.
	 */
	MicroUIGraphicsContext mapMicroUIGraphicsContext(byte[] gc);

	/**
	 * Creates a new {@link MicroUIGraphicsContext} for the given MicroUI GraphicsContext byte array. This allows to
	 * manipulate several destination targets in same native call. However it takes more time to create a new instance
	 * at each native call than mapping the unique instance.
	 * <p>
	 * The implementation assumes given byte array has been retrieved by MicroEJ application calling
	 * <code>gc.getSNIContext()</code>. If given byte array does not respect this rule, the behavior is unknown.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext representation.
	 * @return a new {@link MicroUIGraphicsContext} object updated with the given MicroUI GraphicsContext
	 *         characteristics.
	 *
	 * @see LLUIDisplay#mapMicroUIGraphicsContext(byte[])
	 */
	MicroUIGraphicsContext newMicroUIGraphicsContext(byte[] gc);

	/**
	 * Maps a MicroUI Image byte array (retrieved by MicroEJ application calling <code>image.getSNIContext()</code>) on
	 * a {@link MicroUIImage}. This image can be used as source (copy image in a graphics context) or as target (copy
	 * image in another image).
	 * <p>
	 * Implementation notes:
	 * <ul>
	 * <li>Same object is always returned, only internal {@link MicroUIImage} fields are updated. By consequence this
	 * method can only be called once during a MicroUI native call. See {@link #newMicroUIImage(byte[])} to force to
	 * create a new {@link MicroUIImage} instead.</li>
	 * <li>The implementation assumes given byte array has been retrieved by MicroEJ application calling
	 * <code>image.getSNIContext()</code>. If given byte array does not respect this rule, the behavior is unknown.</li>
	 * </ul>
	 *
	 * @param image
	 *            the MicroUI Image representation.
	 * @return the unique {@link MicroUIImage} object updated with the given MicroUI Image characteristics.
	 */
	MicroUIImage mapMicroUIImage(byte[] image);

	/**
	 * Creates a new {@link MicroUIImage} for the given MicroUI Image byte array. This allows to manipulate several
	 * sources and targets in same native call. However it takes more time to create a new instance at each native call
	 * than mapping the unique instance.
	 * <p>
	 * The implementation assumes given byte array has been retrieved by MicroEJ application calling
	 * <code>image.getSNIContext()</code>. If given byte array does not respect this rule, the behavior is unknown.
	 *
	 * @param image
	 *            the MicroUI Image representation.
	 * @return a new {@link MicroUIImage} object updated with the given MicroUI Image characteristics.
	 *
	 * @see LLUIDisplay#mapMicroUIImage(byte[])
	 */
	MicroUIImage newMicroUIImage(byte[] image);

	/**
	 * Gets the number of bits (opacity included) to represent a pixel on the hardware device. For instance a RGB565
	 * display returns 16.
	 *
	 * @return the hardware device pixel depth.
	 */
	int getDisplayPixelDepth();

	/**
	 * Requests a call to <code>Display.flush()</code>. The call of <code>Display.flush()</code> is synchronized with
	 * the MicroEJ application drawing (see MicroUI event pump).
	 *
	 * @param force
	 *            true to force a call to <code>Display.flush()</code> even if the application dirty area is null (the
	 *            flush seems actually useless); false to not call <code>Display.flush()</code> if the the application
	 *            dirty area is null.
	 */
	void requestFlush(boolean force);

	/**
	 * Requests a call to <code>Displayable.render()</code> (current displayable on display). The call of
	 * <code>Displayable.render()</code> is synchronized with the MicroEJ application drawing (see MicroUI event pump).
	 */
	void requestRender();

	/**
	 * Converts the given MicroUI color (format 0xAARRGGBB) in a truncated color (same format: 0xAARRGGBB).
	 * <p>
	 * This color takes in consideration the hardware display pixel representation constraints. For instance, for a
	 * display with the pixel format RGB565, the truncated color of green color (0xff00ff00) is 0xff00fc00 (because
	 * RGB565 format keeps only 6 first bits for green component).
	 * <p>
	 * This truncated color must be used to convert the MicroUI GraphicsContext current color in order to respect the
	 * hardware display pixel representation constraints.
	 *
	 * @param color
	 *            the MicroUI color in format 0xAARRGGBB.
	 * @return the truncated color in format 0xAARRGGBB.
	 */
	int convertARGBColorToColorToDraw(int color);

	/**
	 * Converts the colors of the pixels in the given image's region.
	 *
	 * This is useful when the given region has been drawn by a third-party algorithm that doesn't respect the image
	 * format restrictions. For instance an anti-aliased drawing can draw a pixel with the color 0xfff3f3f3. On a RGB565
	 * image, this color must be converted in 0xfff0f0f0.
	 *
	 * The implementation consists to call {@link LLUIDisplay#convertARGBColorToColorToDraw(int)} for each region's
	 * pixel.
	 *
	 * @param image
	 *            the image to update.
	 * @param x
	 *            the top-left pixel X coordinate.
	 * @param y
	 *            the top-left pixel Y coordinate.
	 * @param width
	 *            the region width.
	 * @param height
	 *            the region height.
	 */
	void convertRegion(MicroUIImage image, int x, int y, int width, int height);

	/**
	 * Blends two colors applying a global alpha factor.
	 *
	 * @param foreground
	 *            the ARGB8888 foreground color.
	 * @param background
	 *            the ARGB8888 background color.
	 * @param alpha
	 *            the global alpha factor.
	 *
	 * @return an ARGB8888 color.
	 */
	int blend(int foreground, int background, int alpha);

	/**
	 * Tells if format is a custom format or not.
	 *
	 * @see MicroUIImageFormat
	 *
	 * @param format
	 *            the format to check.
	 *
	 * @return true if format refers to a custom format
	 */
	boolean isCustomFormat(MicroUIImageFormat format);

}
