/*
 * Java
 *
 * Copyright 2021-2024 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.microvg.test;

import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

import ej.bon.Constants;
import ej.microui.display.Colors;
import ej.microui.display.Display;
import ej.microui.display.GraphicsContext;
import ej.microvg.Matrix;
import ej.microvg.Path;
import ej.microvg.VectorGraphicsPainter;

/**
 * Tests the drawing of colored path with matrix transformations.
 */
@SuppressWarnings("nls")
public class TestMatrix {

	/**
	 * Starts MicroUI.
	 */
	@BeforeClass
	public static void pre() {
		TestUtilities.startMicroUI();
	}

	/**
	 * Stops MicroUI.
	 */
	@AfterClass
	public static void post() {
		TestUtilities.stopMicroUI();
	}

	/**
	 * Resets the content of the screen to black.
	 */
	@Before
	public static void preTest() {
		TestUtilities.clearScreen();
	}

	/**
	 * Tests a square path.
	 */
	@Test
	public static void testMatrixIdentity() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.moveTo(0, 0);
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);
		path.close();

		g.setColor(Colors.RED);
		VectorGraphicsPainter.fillPath(g, path, new Matrix());
		display.flush();

		TestUtilities.check("top left", 0, 0, g, Colors.RED);
		TestUtilities.check("center", 49, 49, g, Colors.RED);
		TestUtilities.check("bottom right", 99, 99, g, Colors.RED);
	}

	/**
	 * Tests a translated square path.
	 */
	@Test
	public static void testMatrixTranslate() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.moveTo(0, 0);
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);
		path.close();

		Matrix matrix = new Matrix();
		matrix.preTranslate(100, 0);

		g.setColor(Colors.RED);
		VectorGraphicsPainter.fillPath(g, path, matrix);
		display.flush();

		TestUtilities.check("top left", 100, 0, g, Colors.RED);
		TestUtilities.check("center", 149, 49, g, Colors.RED);
		TestUtilities.check("bottom right", 199, 99, g, Colors.RED);
	}

	/**
	 * Tests a rotated square path.
	 */
	@Test
	public static void testMatrixRotate() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.moveTo(0, 0);
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);
		path.close();

		Matrix matrix = new Matrix();
		matrix.setRotate(45);

		g.setColor(Colors.RED);
		VectorGraphicsPainter.fillPath(g, path, matrix);
		display.flush();

		TestUtilities.check("top left", 0, 1, g, Colors.RED);
		TestUtilities.check("center", 0, 70, g, Colors.RED);
		TestUtilities.check("top right", 69, 70, g, Colors.RED);
		TestUtilities.check("bottom right", 0, 139, g, Colors.RED);
	}

	/**
	 * Tests a translated and rotated square path.
	 */
	@Test
	public static void testMatrixTranslateRotate() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.moveTo(0, 0);
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);
		path.close();

		Matrix matrix = new Matrix();
		matrix.setTranslate(100, 0);
		matrix.preRotate(45);

		g.setColor(Colors.RED);
		VectorGraphicsPainter.fillPath(g, path, matrix);
		display.flush();

		TestUtilities.check("top left", 100, 1, g, Colors.RED);
		TestUtilities.check("center", 100, 70, g, Colors.RED);
		TestUtilities.check("top right", 169, 70, g, Colors.RED);
		TestUtilities.check("bottom right", 100, 139, g, Colors.RED);
	}

	/**
	 * Tests a translated and scaled square path.
	 */
	@Test
	public static void testMatrixScaleTranslate() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.moveTo(0, 0);
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);
		path.close();

		Matrix matrix = new Matrix();
		matrix.setTranslate(10, 0);
		matrix.preScale(3f, 2f);

		g.setColor(Colors.RED);
		VectorGraphicsPainter.fillPath(g, path, matrix);
		display.flush();

		TestUtilities.check("top left", 10, 0, g, Colors.RED);
		TestUtilities.check("center", 159, 99, g, Colors.RED);
		TestUtilities.check("bottom right", 309, 199, g, Colors.RED);
	}

	/**
	 * Tests square path with several transformations.
	 */
	@Test
	public static void testMatrixComplex() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.moveTo(0, 0);
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);
		path.close();

		Matrix matrix = new Matrix();
		matrix.setTranslate(100, 0);
		matrix.preScale(2f, .5f);
		matrix.preRotate(45);
		matrix.postScale(.5f, 2f);
		matrix.postRotate(-45);
		matrix.postTranslate(0, 100);

		g.setColor(Colors.RED);
		VectorGraphicsPainter.fillPath(g, path, matrix);
		display.flush();

		TestUtilities.check("top left", 36, 66, g, Colors.RED);
		TestUtilities.check("center", 84, 114, g, Colors.RED);
		TestUtilities.check("bottom right", 133, 163, g, Colors.RED);
	}

	/**
	 * Tests square path with a matrix set in another matrix.
	 */
	@Test
	public static void testMatrixSet() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.moveTo(0, 0);
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);
		path.close();

		Matrix matrix1 = new Matrix();
		matrix1.setTranslate(100, 0);
		matrix1.preScale(2f, .5f);
		matrix1.preRotate(45);

		Matrix matrix2 = new Matrix();
		matrix2.setTranslate(50, 0);
		matrix2.preScale(2f, 2f);
		matrix2.preRotate(30);

		matrix1.set(matrix2);

		g.setColor(Colors.RED);
		VectorGraphicsPainter.fillPath(g, path, matrix1);
		display.flush();

		TestUtilities.check("top left", 50, 1, g, Colors.RED);
		TestUtilities.check("top right", 221, 100, g, Colors.RED);
		TestUtilities.check("center", 87, 136, g, Colors.RED);
		TestUtilities.check("bottom right", 122, 270, g, Colors.RED);
	}

	/**
	 * Tests square path with a matrix created by copying another matrix.
	 */
	@Test
	public static void testMatrixCopy() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.moveTo(0, 0);
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);
		path.close();

		Matrix matrix2 = new Matrix();
		matrix2.setTranslate(50, 0);
		matrix2.preScale(2f, 2f);
		matrix2.preRotate(30);

		Matrix matrix1 = new Matrix(matrix2);

		g.setColor(Colors.RED);
		VectorGraphicsPainter.fillPath(g, path, matrix1);
		display.flush();

		TestUtilities.check("top left", 50, 1, g, Colors.RED);
		TestUtilities.check("top right", 221, 100, g, Colors.RED);
		TestUtilities.check("center", 87, 136, g, Colors.RED);
		TestUtilities.check("bottom right", 122, 270, g, Colors.RED);
	}

	/**
	 * Tests square path with a matrix pre concatenate with another matrix.
	 */
	@Test
	public static void testMatrixPreConcat() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.moveTo(0, 0);
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);
		path.close();

		Matrix matrix1 = new Matrix();
		matrix1.setTranslate(50, 0);

		Matrix matrix2 = new Matrix();
		matrix2.setScale(2f, 2f);
		matrix2.preRotate(30);

		matrix1.preConcat(matrix2);

		g.setColor(Colors.RED);
		VectorGraphicsPainter.fillPath(g, path, matrix1);
		display.flush();

		TestUtilities.check("top left", 50, 1, g, Colors.RED);
		TestUtilities.check("top right", 221, 100, g, Colors.RED);
		TestUtilities.check("center", 87, 136, g, Colors.RED);
		TestUtilities.check("bottom right", 122, 270, g, Colors.RED);
	}

	/**
	 * Tests square path with a matrix post concatenate with another matrix.
	 */
	@Test
	public static void testMatrixPostConcat() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.moveTo(0, 0);
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);
		path.close();

		Matrix matrix1 = new Matrix();
		matrix1.setScale(2f, 2f);
		matrix1.preRotate(30);

		Matrix matrix2 = new Matrix();
		matrix2.setTranslate(50, 0);

		matrix1.postConcat(matrix2);

		g.setColor(Colors.RED);
		VectorGraphicsPainter.fillPath(g, path, matrix1);
		display.flush();

		TestUtilities.check("top left", 50, 1, g, Colors.RED);
		TestUtilities.check("top right", 221, 100, g, Colors.RED);
		TestUtilities.check("center", 87, 136, g, Colors.RED);
		TestUtilities.check("bottom right", 122, 270, g, Colors.RED);
	}

	/**
	 * Tests square path with a matrix concatenate with another matrix.
	 */
	@Test
	public static void testMatrixSetConcatSelf() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.moveTo(0, 0);
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);
		path.close();

		Matrix matrix1 = new Matrix();
		matrix1.setTranslate(50, 0);

		Matrix matrix2 = new Matrix();
		matrix2.setScale(2f, 2f);
		matrix2.preRotate(30);

		matrix1.setConcat(matrix1, matrix2);

		g.setColor(Colors.RED);
		VectorGraphicsPainter.fillPath(g, path, matrix1);
		display.flush();

		TestUtilities.check("top left", 50, 1, g, Colors.RED);
		TestUtilities.check("top right", 221, 100, g, Colors.RED);
		TestUtilities.check("center", 87, 136, g, Colors.RED);
		TestUtilities.check("bottom right", 122, 270, g, Colors.RED);
	}

	/**
	 * Tests square path with a matrix the result of the concatenation of two other matrices.
	 */
	@Test
	public static void testMatrixSetConcat() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.moveTo(0, 0);
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);
		path.close();

		Matrix matrix1 = new Matrix();
		matrix1.setTranslate(100, 0);
		matrix1.preScale(2f, .5f);
		matrix1.preRotate(45);

		Matrix matrix2 = new Matrix();
		matrix2.setTranslate(50, 0);

		Matrix matrix3 = new Matrix();
		matrix3.setScale(2f, 2f);
		matrix3.preRotate(30);

		matrix1.setConcat(matrix2, matrix3);

		g.setColor(Colors.RED);
		VectorGraphicsPainter.fillPath(g, path, matrix1);
		display.flush();

		TestUtilities.check("top left", 50, 1, g, Colors.RED);
		TestUtilities.check("top right", 221, 100, g, Colors.RED);
		TestUtilities.check("center", 87, 136, g, Colors.RED);
		TestUtilities.check("bottom right", 122, 270, g, Colors.RED);
	}

	/**
	 * Tests square path with a reseted matrix.
	 */
	@Test
	public static void testMatrixReset() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.moveTo(0, 0);
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);
		path.close();

		Matrix matrix1 = new Matrix();
		matrix1.setTranslate(100, 0);
		matrix1.preScale(2f, .5f);
		matrix1.preRotate(45);
		matrix1.reset();
		matrix1.setTranslate(50, 0);
		matrix1.preScale(2f, 2f);
		matrix1.preRotate(30);

		g.setColor(Colors.RED);
		VectorGraphicsPainter.fillPath(g, path, matrix1);
		display.flush();

		TestUtilities.check("top left", 50, 1, g, Colors.RED);
		TestUtilities.check("top right", 221, 100, g, Colors.RED);
		TestUtilities.check("center", 87, 136, g, Colors.RED);
		TestUtilities.check("bottom right", 122, 270, g, Colors.RED);
	}

	/**
	 * Tests {@link Matrix#setConcat(Matrix, Matrix)} with side effects.
	 */
	@Test
	public static void testMatrixSetConcatSideEffect() {

		Matrix matrixA1 = new Matrix();
		matrixA1.setTranslate(50, 0);
		Matrix matrixB1 = new Matrix();
		matrixB1.setScale(2f, 2f);
		matrixB1.preRotate(30);

		Matrix matrixA2 = new Matrix(matrixA1);
		Matrix matrixB2 = new Matrix(matrixB1);

		if (Constants.getBoolean(TestUtilities.DEBUG_CONSTANT)) {
			System.out.println("C = A x B");
		}
		Matrix matrixAxB = new Matrix();
		matrixAxB.setConcat(matrixA1, matrixB1);
		TestUtilities.checkMatrix(matrixA1, matrixA2, true);
		TestUtilities.checkMatrix(matrixB1, matrixB2, true);

		if (Constants.getBoolean(TestUtilities.DEBUG_CONSTANT)) {
			System.out.println("C = B x A");
		}
		Matrix matrixBxA = new Matrix();
		matrixBxA.setConcat(matrixB1, matrixA1);
		TestUtilities.checkMatrix(matrixA1, matrixA2, true);
		TestUtilities.checkMatrix(matrixB1, matrixB2, true);

		if (Constants.getBoolean(TestUtilities.DEBUG_CONSTANT)) {
			System.out.println("A' = A x B");
		}
		matrixA1.setConcat(matrixA1, matrixB1);
		TestUtilities.checkMatrix(matrixA1, matrixAxB, true);
		TestUtilities.checkMatrix(matrixB1, matrixB2, true);
		matrixA1 = new Matrix(matrixA2);

		if (Constants.getBoolean(TestUtilities.DEBUG_CONSTANT)) {
			System.out.println("A' = B x A");
		}
		matrixA1.setConcat(matrixB1, matrixA1);
		TestUtilities.checkMatrix(matrixA1, matrixBxA, true);
		TestUtilities.checkMatrix(matrixB1, matrixB2, true);
		matrixA1 = new Matrix(matrixA2);

		if (Constants.getBoolean(TestUtilities.DEBUG_CONSTANT)) {
			System.out.println("B' = A x B");
		}
		matrixB1.setConcat(matrixA1, matrixB1);
		TestUtilities.checkMatrix(matrixA1, matrixA2, true);
		TestUtilities.checkMatrix(matrixB1, matrixAxB, true);
		matrixB1 = new Matrix(matrixB2);

		if (Constants.getBoolean(TestUtilities.DEBUG_CONSTANT)) {
			System.out.println("B' = B x A");
		}
		matrixB1.setConcat(matrixB1, matrixA1);
		TestUtilities.checkMatrix(matrixA1, matrixA2, true);
		TestUtilities.checkMatrix(matrixB1, matrixBxA, true);
		matrixB1 = new Matrix(matrixB2);
	}

	/**
	 * Tests {@link Matrix#preConcat(Matrix)} with side effects.
	 */
	@Test
	public static void testMatrixPreConcatSideEffect() {

		Matrix matrixA1 = new Matrix();
		matrixA1.setTranslate(50, 0);
		Matrix matrixB1 = new Matrix();
		matrixB1.setScale(2f, 2f);
		matrixB1.preRotate(30);

		Matrix matrixA2 = new Matrix(matrixA1);
		Matrix matrixB2 = new Matrix(matrixB1);

		Matrix matrixAxB = new Matrix();
		matrixAxB.setConcat(matrixA1, matrixB1);
		Matrix matrixBxA = new Matrix();
		matrixBxA.setConcat(matrixB1, matrixA1);

		if (Constants.getBoolean(TestUtilities.DEBUG_CONSTANT)) {
			System.out.println("A' = A x B");
		}
		matrixA1.preConcat(matrixB1);
		TestUtilities.checkMatrix(matrixB1, matrixB2, true);
		TestUtilities.checkMatrix(matrixA1, matrixAxB, true);
		matrixA1 = new Matrix(matrixA2);

		if (Constants.getBoolean(TestUtilities.DEBUG_CONSTANT)) {
			System.out.println("B' = B x A");
		}
		matrixB1.preConcat(matrixA1);
		TestUtilities.checkMatrix(matrixA1, matrixA2, true);
		TestUtilities.checkMatrix(matrixB1, matrixBxA, true);
		matrixB1 = new Matrix(matrixB2);
	}

	/**
	 * Tests {@link Matrix#postConcat(Matrix)} with side effects.
	 */
	@Test
	public static void testMatrixPostConcatSideEffect() {

		Matrix matrixA1 = new Matrix();
		matrixA1.setTranslate(50, 0);
		Matrix matrixB1 = new Matrix();
		matrixB1.setScale(2f, 2f);
		matrixB1.preRotate(30);

		Matrix matrixA2 = new Matrix(matrixA1);
		Matrix matrixB2 = new Matrix(matrixB1);

		Matrix matrixAxB = new Matrix();
		matrixAxB.setConcat(matrixA1, matrixB1);
		Matrix matrixBxA = new Matrix();
		matrixBxA.setConcat(matrixB1, matrixA1);

		if (Constants.getBoolean(TestUtilities.DEBUG_CONSTANT)) {
			System.out.println("A' = B x A");
		}
		matrixA1.postConcat(matrixB1);
		TestUtilities.checkMatrix(matrixB1, matrixB2, true);
		TestUtilities.checkMatrix(matrixA1, matrixBxA, true);
		matrixA1 = new Matrix(matrixA2);

		if (Constants.getBoolean(TestUtilities.DEBUG_CONSTANT)) {
			System.out.println("B' = A x B");
		}
		matrixB1.postConcat(matrixA1);
		TestUtilities.checkMatrix(matrixA1, matrixA2, true);
		TestUtilities.checkMatrix(matrixB1, matrixAxB, true);
		matrixB1 = new Matrix(matrixB2);
	}
}
