/*
 * Copyright 2015-2022 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.mwt.style;

import ej.annotation.Nullable;
import ej.microui.display.Font;
import ej.mwt.style.background.Background;
import ej.mwt.style.dimension.Dimension;
import ej.mwt.style.outline.Outline;
import ej.mwt.util.Alignment;

/**
 * A style describes how widgets must be rendered on screen. The attributes of the style are strongly inspired from CSS.
 * <p>
 * The dimension is used to control the size of the widget.
 * <p>
 * The horizontal and vertical alignments are used to position the content of the widget within its bounds.
 * <p>
 * The margin, border and padding are the used to wrap the content of the widget. The widget is wrapped in the following
 * sequence: first the padding, then the border, and finally the margin.<br>
 * <img src="doc-files/boxmodel.png" alt="Outlines model.">
 * <p>
 * The background is used to render the background of the widget.
 * <p>
 * The color can be used to render the content of the widget, such as text or drawings.
 * <p>
 * The font can be used to render the text.
 */
public interface Style {

	/**
	 * Gets the dimension.
	 *
	 * @return the dimension.
	 */
	Dimension getDimension();

	/**
	 * Gets the horizontal alignment.
	 *
	 * @return the horizontal alignment.
	 * @see Alignment
	 */
	int getHorizontalAlignment();

	/**
	 * Gets the vertical alignment.
	 *
	 * @return the vertical alignment.
	 * @see Alignment
	 */
	int getVerticalAlignment();

	/**
	 * Gets the margin.
	 *
	 * @return the margin.
	 */
	Outline getMargin();

	/**
	 * Gets the border.
	 *
	 * @return the border.
	 */
	Outline getBorder();

	/**
	 * Gets the padding.
	 *
	 * @return the padding.
	 */
	Outline getPadding();

	/**
	 * Gets the background.
	 *
	 * @return the background.
	 */
	Background getBackground();

	/**
	 * Gets the color.
	 *
	 * @return the color.
	 */
	int getColor();

	/**
	 * Gets the font.
	 *
	 * @return the font.
	 */
	Font getFont();

	/**
	 * Gets an extra field as an object of the given type.
	 *
	 * @param fieldId
	 *            the ID of the extra field.
	 * @param clazz
	 *            the type of the extra field.
	 * @param defaultValue
	 *            the value to return if the extra field is not set or if it does not match the given type.
	 * @param <T>
	 *            the type of the extra field.
	 * @return the value of the extra field.
	 */
	<T> T getExtraObject(int fieldId, Class<T> clazz, T defaultValue);

	/**
	 * Gets an extra field as an int.
	 *
	 * @param fieldId
	 *            the ID of the extra field.
	 * @param defaultValue
	 *            the value to return if the extra field is not set or if it is not an integer.
	 * @return the value of the extra field.
	 */
	int getExtraInt(int fieldId, int defaultValue);

	/**
	 * Gets an extra field as a float.
	 *
	 * @param fieldId
	 *            the ID of the extra field.
	 * @param defaultValue
	 *            the value to return if the extra field is not set or if it is not a float.
	 * @return the value of the extra field.
	 */
	float getExtraFloat(int fieldId, float defaultValue);

	// define this method to avoid embedding the equals() method of every class (M0081MEJA-1240)
	@Override
	boolean equals(@Nullable Object obj);

	// define this method to avoid embedding the hashCode() method of every class (M0081MEJA-1240)
	@Override
	int hashCode();
}
