/*
 * Java
 *
 * Copyright 2019-2020 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microui.event;

/**
 * This class holds all methods to send MicroUI Touch events to MicroUI application.
 */
@SuppressWarnings("nls")
public class EventTouch {

	/**
	 * Usual MicroUI Touch generator tag used in platform configuration project (<code>microui.xml</code>).
	 */
	public static final String COMMON_MICROUI_GENERATOR_TAG = "TOUCH";

	private EventTouch() {
		// hide implicit constructor
	}

	/**
	 * Send a PRESSED event to the event generator with the MicroUI generator tag {@link #COMMON_MICROUI_GENERATOR_TAG}.
	 *
	 * @param x
	 *            the pointer X coordinate.
	 * @param y
	 *            the pointer Y coordinate.
	 *
	 * @see #sendPressedEvent(String, int, int)
	 */
	public static void sendPressedEvent(int x, int y) {
		sendPressedEvent(COMMON_MICROUI_GENERATOR_TAG, x, y);
	}

	/**
	 * Send a PRESSED event to the event generator.
	 * <p>
	 * The result of this method is unknown when the given event generator is not a MicroUI <code>Touch</code> event
	 * generator.
	 *
	 * @param touchEventGenerator
	 *            the MicroUI <code>Touch</code> event generator.
	 * @param x
	 *            the pointer X coordinate.
	 * @param y
	 *            the pointer Y coordinate.
	 *
	 */
	public static void sendPressedEvent(String touchEventGenerator, int x, int y) {
		EventPointer.sendPressedEvent(touchEventGenerator, 0, x, y, true);
	}

	/**
	 * Send a RELEASED event to the event generator with the MicroUI generator tag
	 * {@link #COMMON_MICROUI_GENERATOR_TAG}.
	 *
	 * @see #sendReleasedEvent(String)
	 */
	public static void sendReleasedEvent() {
		sendReleasedEvent(COMMON_MICROUI_GENERATOR_TAG);
	}

	/**
	 * Send a RELEASED event to the event generator.
	 * <p>
	 * The result of this method is unknown when the given event generator is not a MicroUI <code>Pointer</code> event
	 * generator.
	 *
	 * @param touchEventGenerator
	 *            the MicroUI <code>Touch</code> event generator.
	 *
	 */
	public static void sendReleasedEvent(String touchEventGenerator) {
		EventPointer.sendReleasedEvent(touchEventGenerator, 0);
	}

	/**
	 * Send a MOVE event to the event generator with the MicroUI generator tag {@link #COMMON_MICROUI_GENERATOR_TAG}.
	 *
	 * @param x
	 *            the pointer X coordinate.
	 * @param y
	 *            the pointer Y coordinate.
	 *
	 * @see #sendMovedEvent(String, int, int)
	 */
	public static void sendMovedEvent(int x, int y) {
		sendMovedEvent(COMMON_MICROUI_GENERATOR_TAG, x, y);
	}

	/**
	 * Send a MOVE event to the event generator.
	 * <p>
	 * The result of this method is unknown when the given event generator is not a MicroUI <code>Touch</code> event
	 * generator.
	 *
	 * @param touchEventGenerator
	 *            the MicroUI <code>Pointer</code> event generator.
	 * @param x
	 *            the pointer X coordinate.
	 * @param y
	 *            the pointer Y coordinate.
	 *
	 */
	public static void sendMovedEvent(String touchEventGenerator, int x, int y) {
		EventPointer.sendMovedEvent(touchEventGenerator, x, y, true);
	}
}
