/*
 * Java
 *
 * Copyright 2008-2019 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.fp;

/**
 * A widget has to implement this interface to be able to receive user mouse events.
 * <p>
 * All methods does nothing by default which allows the widget implements only the wanted mouse action(s).
 * <p>
 * The given coordinates are widget relative: it is not the main mouse coordinates.
 */
public interface MouseListener {

	/**
	 * Some mouse events hold a mouse button. This enumeration provides an abstraction of mouse buttons identifiers.
	 */
	enum MouseButton {

		/**
		 * Value for first mouse button; often the left one.
		 */
		FIRST_BUTTON,

		/**
		 * Value for second mouse button; often the middle one.
		 */
		SECOND_BUTTON,

		/**
		 * Value for third mouse button; often the right one.
		 */
		THIRD_BUTTON,

		/**
		 * Value for all others mouse buttons.
		 */
		OTHER_BUTTON;
	}

	/**
	 * Invoked when a mouse enters a widget.
	 *
	 * @param x
	 *            the mouse x coordinate relative to widget position.
	 * @param y
	 *            the mouse y coordinate relative to widget position.
	 */
	default void mouseEntered(int x, int y) {
		// does nothing by default
	}

	/**
	 * Invoked when a mouse exits a widget.
	 *
	 * @param x
	 *            the mouse x coordinate relative to widget position.
	 * @param y
	 *            the mouse y coordinate relative to widget position.
	 */
	default void mouseExited(int x, int y) {
		// does nothing by default
	}

	/**
	 * Invoked when a mouse has been moved on a widget.
	 *
	 * @param x
	 *            the mouse x coordinate relative to widget position.
	 * @param y
	 *            the mouse y coordinate relative to widget position.
	 */
	default void mouseMoved(int x, int y) {
		// does nothing by default
	}

	/**
	 * Invoked when a mouse has been dragged on a widget.
	 *
	 * @param x
	 *            the mouse x coordinate relative to widget position.
	 * @param y
	 *            the mouse y coordinate relative to widget position.
	 */
	default void mouseDragged(int x, int y) {
		// does nothing by default
	}

	/**
	 * Invoked when a mouse has been pressed on a widget.
	 *
	 * @param x
	 *            the mouse x coordinate relative to widget position.
	 * @param y
	 *            the mouse y coordinate relative to widget position.
	 * @param button
	 *            the button that has been pressed.
	 */
	default void mousePressed(int x, int y, MouseButton button) {
		// does nothing by default
	}

	/**
	 * Invoked when a mouse has been released on a widget.
	 *
	 * @param x
	 *            the mouse x coordinate relative to widget position.
	 * @param y
	 *            the mouse y coordinate relative to widget position.
	 * @param button
	 *            the button that has been released.
	 */
	default void mouseReleased(int x, int y, MouseButton button) {
		// does nothing by default
	}

	/**
	 * Invoked when a mouse wheel has been moved on a widget.
	 *
	 * @param x
	 *            the mouse x coordinate relative to widget position.
	 * @param y
	 *            the mouse y coordinate relative to widget position.
	 * @param button
	 *            the button that has been wheeled.
	 * @param count
	 *            the number of wheel moves (can be positive or negative according to the way of the wheel).
	 */
	default void mouseWheelMoved(int x, int y, MouseButton button, int count) {
		// does nothing by default
	}
}
