/*
 * Java
 *
 * Copyright 2018-2021 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.util.message;

/**
 * A message logger logs simple messages based on integer IDs to avoid using Strings.
 */
public interface MessageLogger {

	/**
	 * Gets the message builder used to build the messages.
	 *
	 * @return the message builder.
	 */
	public MessageBuilder getMessageBuilder();

	/**
	 * Sets the message builder used to build the messages.
	 *
	 * @param builder
	 *            the message builder to set, cannot be <code>null</code>.
	 */
	public void setMessageBuilder(MessageBuilder builder);

	/**
	 * Logs a message.
	 *
	 * @param level
	 *            the level of the message.
	 * @param category
	 *            the category of the message.
	 * @param id
	 *            the ID of the message.
	 * @throws NullPointerException
	 *             if the given category is <code>null</code>.
	 *
	 * @see MessageBuilder#buildMessage(char, String, int)
	 */
	void log(char level, String category, int id);

	/**
	 * Logs a message with a throwable.
	 *
	 * @param level
	 *            the level of the message.
	 * @param category
	 *            the category of the message.
	 * @param id
	 *            the ID of the message.
	 * @param t
	 *            the throwable.
	 * @throws NullPointerException
	 *             if the given category is <code>null</code>.
	 * @see MessageBuilder#buildMessage(char, String, int, Throwable)
	 */
	void log(char level, String category, int id, Throwable t);

	/**
	 * Logs a message with arguments.
	 *
	 * @param level
	 *            the level of the message.
	 * @param category
	 *            the category of the message.
	 * @param id
	 *            the ID of the message.
	 * @param arguments
	 *            the arguments.
	 * @throws NullPointerException
	 *             if the given category is <code>null</code>.
	 * @see MessageBuilder#buildMessage(char, String, int, Object...)
	 */
	void log(char level, String category, int id, Object... arguments);

	/**
	 * Logs a message with a throwable and arguments.
	 *
	 * @param level
	 *            the level of the message.
	 * @param category
	 *            the category of the message.
	 * @param id
	 *            the ID of the message.
	 * @param t
	 *            the throwable.
	 * @param arguments
	 *            the arguments.
	 * @throws NullPointerException
	 *             if the given category is <code>null</code>.
	 * @see MessageBuilder#buildMessage(char, String, int, Throwable, Object...)
	 */
	void log(char level, String category, int id, Throwable t, Object... arguments);
}
