/*
 * Java
 *
 * Copyright 2018-2025 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.kf.util.security;

import java.security.Permission;

import com.microej.kf.util.message.KfUtilMessage;

import ej.kf.Feature;
import ej.util.message.Level;
import ej.util.message.MessageLogger;

/**
 * Helper implementation of the {@link FeaturePermissionCheckDelegate} that only logs the permission accesses.
 */
public class LoggingPermissionCheckDelegate implements FeaturePermissionCheckDelegate {

	final MessageLogger logger;
	final char logLevel;

	/**
	 * Creates a {@link LoggingPermissionCheckDelegate} with the specified {@link MessageLogger} and {@link Level}
	 *
	 * @param logger
	 *            the logger that should be used to log permission checks
	 * @param logLevel
	 *            the level that should be used to log permission checks
	 */
	public LoggingPermissionCheckDelegate(MessageLogger logger, char logLevel) {
		this.logger = logger;
		this.logLevel = logLevel;
	}

	/**
	 * Creates a {@link LoggingPermissionCheckDelegate} using the KF utilities logger and the FINER log level.
	 */
	public LoggingPermissionCheckDelegate() {
		this(KfUtilMessage.LOGGER, Level.FINER);
	}

	@Override
	public void checkPermission(Permission p, Feature f) {
		logger.log(this.logLevel, KfUtilMessage.CATEGORY, KfUtilMessage.PERMISSION_CHECK,
				f.getName() + " checked for " + p);
	}
}