/*
 * Java
 *
 * Copyright 2025 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.kf.util.monitoring;

import ej.annotation.Nullable;
import ej.basictool.map.PackedMap;
import ej.bon.Timer;
import ej.bon.TimerTask;
import ej.kf.Feature;
import ej.kf.FeatureStateListener;
import ej.kf.Kernel;
import ej.kf.Module;

/**
 * This class represents the implementation for the ResourceMonitorService.
 */
public class ResourceMonitoringServiceImpl implements ResourceMonitoringService, FeatureStateListener {
    private static final int UNLIMITED_QUOTA = Integer.MAX_VALUE;
    private static final int NO_QUOTA = -1;

    private final Timer timer;
    private final long intervalMS;
    private final boolean runGcBeforeCollecting;
    private final MonitoringData monitoringData;
    private final long maxCpuLoadPerMonitoringPeriod;

    @Nullable
    private TimerTask monitoringTask;
    private boolean running;

    /**
     * Instantiates the resource monitoring service.
     *
     * @param timer
     *          the timer
     * @param intervalMS
     *          the timer interval read from the kernel properties
     * @param runGcBeforeCollecting
     *          the boolean read from kernel properties to run GC before collecting monitoring data
     */
    public ResourceMonitoringServiceImpl(Timer timer, long intervalMS, boolean runGcBeforeCollecting) {
        this.timer = timer;
        this.intervalMS = intervalMS;
        this.runGcBeforeCollecting = runGcBeforeCollecting;

        // retrieves value from kernel properties, must be set after calibration
        long maxCpuPerSecond = Long.getLong("monitoring.check.cpu.calibration", 731010); // default for RT1170

        this.maxCpuLoadPerMonitoringPeriod = maxCpuPerSecond * intervalMS / 1000;

        Kernel.addFeatureStateListener(this);

        this.monitoringData = new MonitoringData(runGcBeforeCollecting);
        this.monitoringTask = null;
    }

    /**
     * Starts the resource monitoring task by enabling monitoring on each module.
     */
    public void start() {
        if (running) {
            return; // monitoring task already running
        }

        enableModuleMonitoring(Kernel.getInstance());
        for (Feature module : Kernel.getAllLoadedFeatures()) {
            if (Feature.State.STARTED.equals(module.getState())) {
                enableModuleMonitoring(module);
            }
        }

        monitoringTask = new MonitoringTask(monitoringData, runGcBeforeCollecting);
        timer.scheduleAtFixedRate(monitoringTask, 0, intervalMS);

        running = true;
    }

    /**
     * Stops resource monitoring by disabling monitoring on each module.
     */
    public void stop() {
        if (!running) {
            return;
        }

        monitoringTask.cancel();

        disableModuleMonitoring(Kernel.getInstance());
        for (Feature module : Kernel.getAllLoadedFeatures()) {
            if (Feature.State.STARTED.equals(module.getState())) {
                disableModuleMonitoring(module);
            }
        }

        running = false;
    }

    @Override
    public void stateChanged(Feature app, @Nullable Feature.State state) {
        final Feature.State currentState = app.getState();
        if (Feature.State.STARTED.equals( currentState)) {
            enableModuleMonitoring(app);
        } else if (Feature.State.STOPPED.equals(currentState)) {
            disableModuleMonitoring(app);
        }
    }

    @Override
    public PackedMap<String, Float> getUsedCpuPercentPerModule() {
        PackedMap<String, Float> result = new PackedMap<>();
        synchronized (monitoringData) {
            for (Module module : monitoringData.getExecutionCounterPerModuleMap().keySet()) {
                MonitoringData.ExecutionCounter execCounter = monitoringData.getExecutionCounterPerModuleMap().get(module);
                float percentCpu = (float) (execCounter.getCurrent() - execCounter.getLast())
                        / maxCpuLoadPerMonitoringPeriod;
                result.put(module.getName(), percentCpu);
            }
        }

        return result;
    }

    @Override
    public PackedMap<String, Long> getUsedMemoryPerModule() {
        PackedMap<String, Long> result = new PackedMap<>();
        synchronized (monitoringData) {
            for (Module module : monitoringData.getRamUsagePerModuleMap().keySet()) {
                result.put(module.getName(), monitoringData.getRamUsagePerModuleMap().get(module));
            }
        }

        return result;
    }

    @Override
    public PackedMap<String, Float> getUsedMemoryPercentPerModule() {
        PackedMap<String, Float> result = new PackedMap<>();
        synchronized (monitoringData) {
            for (Module module : monitoringData.getRamUsagePerModuleMap().keySet()) {
                long value = monitoringData.getRamUsagePerModuleMap().get(module);
                result.put(module.getName(), (float) (value) / monitoringData.getTotalMemory());
            }
        }
        return result;
    }

    @Override
    public long getFreeMemory() {
        return monitoringData.getFreeMemory();
    }

    @Override
    public long getMaxMemory() {
        return monitoringData.getMaxMemory();
    }

    @Override
    public long getTotalMemory() {
        return monitoringData.getTotalMemory();
    }

    /**
     * Enables monitoring for a given module (kernel or feature).
     * Adds the initial value for the module in the monitoringData map.
     * Module must not have a quota before enabling monitoring.
     *
     * @param module
     *          the module to monitor
     */
    private void enableModuleMonitoring(Module module) {
        int previousQuota = module.getExecutionQuota();
        boolean noQuota = previousQuota == NO_QUOTA;
        if (noQuota) {
            module.setExecutionQuota(UNLIMITED_QUOTA); // Activate cpu monitoring
            synchronized (monitoringData) {
                monitoringData.getOriginalQuotaPerModuleMap().put(module, previousQuota);
                monitoringData.getExecutionCounterPerModuleMap().put(module, new MonitoringData.ExecutionCounter(0, 0));
                monitoringData.getRamUsagePerModuleMap().put(module, 0L);
            }
        }
    }

    /**
     * Disables monitoring for a given module (kernel or feature).
     * Removes related data from the monitoringData map.
     *
     * @param module
     *          the module to stop monitoring
     */
    private void disableModuleMonitoring(Module module) {
        synchronized (monitoringData) {
            monitoringData.getExecutionCounterPerModuleMap().remove(module);
            monitoringData.getRamUsagePerModuleMap().remove(module);
            Integer originalQuota = monitoringData.getOriginalQuotaPerModuleMap().remove(module);
            if (originalQuota != null) {
                module.setExecutionQuota(originalQuota);
            }
        }
    }

    /**
     * This class represents the monitoring task ran at regular interval.
     */
    public static final class MonitoringTask extends TimerTask {

        private final MonitoringData monitoringData;
        private final boolean runGcBeforeCollecting;

        /**
         * Instantiates the monitoring task.
         *
         * @param monitoringData
         *          the data structure to regroup all the monitoring data
         * @param runGcBeforeCollecting
         *          the boolean to call the GC before collecting new monitoring data
         */
        public MonitoringTask(final MonitoringData monitoringData, boolean runGcBeforeCollecting) {
            this.monitoringData = monitoringData;
            this.runGcBeforeCollecting = runGcBeforeCollecting;
        }

        @Override
        public void run() {
            if (runGcBeforeCollecting) {
                Runtime.getRuntime().gc(); // NOSONAR Call used to increase heap monitoring accuracy
            }

            for (Feature app : Kernel.getAllLoadedFeatures()) {
                if (app.getState() == Feature.State.STARTED) {
                    computeModuleCpuUsage(app);
                    computeModuleMemoryUsage(app);
                }
            }

            computeModuleCpuUsage(Kernel.getInstance());
            computeModuleMemoryUsage(Kernel.getInstance());
        }

        /**
         * Retrieves and updates CPU usage for the given module.
         *
         * @param module
         *          the module to monitor
         */
        private void computeModuleCpuUsage(Module module) {
            synchronized (monitoringData) {
                MonitoringData.ExecutionCounter exec = monitoringData.getExecutionCounterPerModuleMap().get(module);
                long newExecutionCounter = module.getExecutionCounter();
                // if new exec counter is less than last. exec counter was reset
                exec.setLast(newExecutionCounter < exec.getCurrent() ? 0 : exec.getCurrent());
                exec.setCurrent(newExecutionCounter);
            }
        }

        /**
         * Retrieves and updates Memory usage for the given module.
         *
         * @param module
         *          the module to monitor
         */
        private void computeModuleMemoryUsage(Module module) {
            synchronized (monitoringData) {
                monitoringData.getRamUsagePerModuleMap().put(module, module.getAllocatedMemory());
            }
        }
    }

}
