/*
 * Java
 *
 * Copyright 2024-2025 MicroEJ Corp.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.microvg.test;

import org.junit.Assert;
import org.junit.Test;

import ej.microui.MicroUI;
import ej.microui.MicroUIException;
import ej.microvg.BlendMode;
import ej.microvg.BufferedVectorImage;
import ej.microvg.LinearGradient;
import ej.microvg.Matrix;
import ej.microvg.Path;
import ej.microvg.ResourceVectorImage;
import ej.microvg.VectorFont;
import ej.microvg.VectorGraphicsPainter.FillType;
import ej.microvg.VectorImage;
import ej.microvg.VectorImageBuilder;

/**
 * Tests that MicroVG cannot be used until MicroUI is started: one test per MicroVG entry point.
 */
@SuppressWarnings("nls")
public class TestStartup {

	private static final String MASCOT = "/com/microej/microvg/test/mascot.xml";

	/**
	 * Tests using blend modes
	 */
	@Test
	public static void testBlendMode() {
		runTest(new Runnable() {
			@Override
			public void run() {
				BlendMode.DST_IN.compareTo(BlendMode.DST_OUT);
			}
		}, false); // MicroUI is not required
	}

	/**
	 * Tests creating a buffered image
	 */
	@Test
	public static void testBufferedImage() {
		runTest(new Runnable() {
			@SuppressWarnings({ "resource", "unused" })
			@Override
			public void run() {
				new BufferedVectorImage(10, 10);
			}
		}, true); // MicroUI BufferedImage initialization is required
	}

	/**
	 * Tests creating and using a linear gradient
	 */
	@Test
	public static void testLinearGradient() {
		runTest(new Runnable() {
			@SuppressWarnings("unused")
			@Override
			public void run() {
				new LinearGradient(0, 0, 1, 1, new int[] { 0 }).getMatrix();
			}
		}, false); // MicroUI is not required
	}

	/**
	 * Tests creating and using a matrix
	 */
	@Test
	public static void testMatrix() {
		runTest(new Runnable() {
			@Override
			public void run() {
				new Matrix().reset();
			}
		}, false); // MicroUI is not required
	}

	/**
	 * Tests creating and using a path
	 */
	@Test
	public static void testPath() {
		runTest(new Runnable() {
			@SuppressWarnings("unused")
			@Override
			public void run() {
				new Path().getRightBound();
			}
		}, false); // MicroUI is not required
	}

	/**
	 * Tests creating and using a resource image
	 */
	@Test
	public static void testResourceImage() {
		runTest(new Runnable() {
			@SuppressWarnings({ "resource", "unused" })
			@Override
			public void run() {
				ResourceVectorImage.loadImage(MASCOT).getHeight();
			}
		}, false); // MicroUI is not required
	}

	/**
	 * Tests creating and using a font
	 */
	@Test
	public static void testFont() {
		runTest(new Runnable() {
			@SuppressWarnings({ "resource", "unused" })
			@Override
			public void run() {
				VectorFont.loadFont("/fonts/firstfont.ttf");
			}
		}, false); // MicroUI is not required
	}

	/**
	 * Tests creating and using a raw image
	 */
	@Test
	public static void testRawImage() {
		runTest(new Runnable() {
			@SuppressWarnings({ "resource", "unused" })
			@Override
			public void run() {
				VectorImage.getImage(MASCOT).getWidth();
			}
		}, false); // MicroUI is not required
	}

	/**
	 * Tests creating and using an image builder
	 */
	@Test
	public static void testImageBuilder() {
		runTest(new Runnable() {
			@SuppressWarnings({ "resource", "unused" })
			@Override
			public void run() {
				new VectorImageBuilder(10, 10).addPath(new Path(), 0, FillType.EVEN_ODD);
			}
		}, false); // MicroUI is not required
	}

	private static void runTest(Runnable r, boolean microuiExceptionExpected) {
		try {
			r.run();
			Assert.assertFalse("MicroUI exception not expected", microuiExceptionExpected);
		} catch (MicroUIException e) {
			Assert.assertTrue("MicroUI exception expected", microuiExceptionExpected);
			Assert.assertEquals("MicroUI 'not started' exception expected", MicroUIException.MICROUI_NOT_STARTED,
					e.getErrorCode());
		}
		Assert.assertFalse("MicroUI is not started (sanity check)", MicroUI.isStarted());
	}

}
