/*
 * Java
 *
 * Copyright 2023-2025 MicroEJ Corp.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.microvg.test;

import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.BeforeClass;
import org.junit.Test;

import ej.microui.display.Colors;
import ej.microui.display.GraphicsContext;
import ej.microvg.BlendMode;
import ej.microvg.BufferedVectorImage;
import ej.microvg.LinearGradient;
import ej.microvg.Matrix;
import ej.microvg.Path;
import ej.microvg.ResourceVectorImage;
import ej.microvg.VectorFont;
import ej.microvg.VectorGraphicsPainter;
import ej.microvg.VectorGraphicsPainter.Direction;
import ej.microvg.VectorGraphicsPainter.FillType;

/**
 * Tests the reporting of out-of-memory errors when using BufferedVectorImages.
 */
@SuppressWarnings("nls")
public class TestBufferedVectorImageOOM {
	private static final int OUT_OF_MEMORY_FLAGS = GraphicsContext.DRAWING_LOG_ERROR
			| GraphicsContext.DRAWING_LOG_LIBRARY_INCIDENT | GraphicsContext.DRAWING_LOG_OUT_OF_MEMORY;
	private static final int ITERATIONS = 20;

	private static final String IMAGE_PATH = "/com/microej/microvg/test/animated_colors.xml";
	private static final String FONT_PATH = "/fonts/firstfont.ttf";
	private static final float FONT_SIZE = 16f;
	private static final long LETTER_SPACING = 0;
	private static final int ALPHA = 128;
	private static final float CIRCLE_RADIUS = 10f;

	/**
	 * Initializes MicroUI.
	 */
	@BeforeClass
	public static void setUpBeforeClass() {
		TestUtilities.startMicroUI();
	}

	/**
	 * Stops MicroUI.
	 */
	@AfterClass
	public static void tearDownAfterClass() {
		try {
			// sleeps after last test to flush the traces
			Thread.sleep(10000);
		} catch (InterruptedException e) {
			// nothing to do
		}
		TestUtilities.stopMicroUI();
	}

	private static Path rectanglePath(float width, float height) {
		Path path = new Path();
		path.moveTo(0f, 0f);
		path.lineTo(0f, height);
		path.lineTo(width, height);
		path.lineTo(width, 0f);
		path.close();
		return path;
	}

	private static float[] shiftColorMatrix() {
		return new float[] { //
				0f, 1f, 0f, 0f, 0f, // New red = old green
				0f, 0f, 1f, 0f, 0f, // New green = old blue
				1f, 0f, 0f, 0f, 0f, // New blue = old red
				0f, 0f, 0f, 1f, 0f // Alpha is left untouched
		};
	}

	private BufferedVectorImage sourceBufferedVectorImage() {
		BufferedVectorImage bvi = new BufferedVectorImage(1, 1);
		VectorGraphicsPainter.fillPath(bvi.getGraphicsContext(), rectanglePath(1f, 1f), 0f, 0f);
		return bvi;
	}

	/**
	 * Tests that adding too many "draw animated image" blocks to a BufferedVectorImage causes the proper error flags to
	 * be set.
	 */
	@Test
	public void testDrawAnimatedImage1() {

		try (BufferedVectorImage bvi = new BufferedVectorImage(1, 1);
				ResourceVectorImage source = ResourceVectorImage.loadImage(IMAGE_PATH)) {
			Matrix matrix = new Matrix();

			for (int i = 0; i < ITERATIONS; i++) {
				VectorGraphicsPainter.drawAnimatedImage(bvi.getGraphicsContext(), source, matrix, 1);
			}

			Assert.assertEquals(OUT_OF_MEMORY_FLAGS, bvi.getGraphicsContext().getAndClearDrawingLogFlags());
		}
	}

	/**
	 * Tests that adding too many "draw animated image" blocks to a BufferedVectorImage causes the proper error flags to
	 * be set.
	 */
	@Test
	public void testDrawAnimatedImage2() {

		try (BufferedVectorImage bvi = new BufferedVectorImage(1, 1);
				ResourceVectorImage source = ResourceVectorImage.loadImage(IMAGE_PATH)) {
			for (int i = 0; i < ITERATIONS; i++) {
				VectorGraphicsPainter.drawAnimatedImage(bvi.getGraphicsContext(), source, 0f, 0f, 1);
			}

			Assert.assertEquals(OUT_OF_MEMORY_FLAGS, bvi.getGraphicsContext().getAndClearDrawingLogFlags());
		}
	}

	/**
	 * Tests that adding too many "draw animated image" blocks to a BufferedVectorImage causes the proper error flags to
	 * be set.
	 */
	@Test
	public void testDrawAnimatedImage3() {

		try (BufferedVectorImage bvi = new BufferedVectorImage(1, 1);
				ResourceVectorImage source = ResourceVectorImage.loadImage(IMAGE_PATH)) {
			Matrix matrix = new Matrix();

			for (int i = 0; i < ITERATIONS; i++) {
				VectorGraphicsPainter.drawAnimatedImage(bvi.getGraphicsContext(), source, matrix, 1, ALPHA);
			}

			Assert.assertEquals(OUT_OF_MEMORY_FLAGS, bvi.getGraphicsContext().getAndClearDrawingLogFlags());
		}
	}

	/**
	 * Tests that adding too many "draw filtered animated image" blocks to a BufferedVectorImage causes the proper error
	 * flags to be set.
	 */
	@Test
	public void testDrawFilteredAnimatedImage() {

		try (BufferedVectorImage bvi = new BufferedVectorImage(1, 1);
				ResourceVectorImage source = ResourceVectorImage.loadImage(IMAGE_PATH)) {
			Matrix matrix = new Matrix();

			for (int i = 0; i < ITERATIONS; i++) {
				VectorGraphicsPainter.drawFilteredAnimatedImage(bvi.getGraphicsContext(), source, matrix, 1,
						shiftColorMatrix());
			}

			Assert.assertEquals(OUT_OF_MEMORY_FLAGS, bvi.getGraphicsContext().getAndClearDrawingLogFlags());
		}
	}

	/**
	 * Tests that adding too many "draw filtered image" blocks to a BufferedVectorImage causes the proper error flags to
	 * be set.
	 */
	@Test
	public void testDrawFilteredImage() {

		try (BufferedVectorImage bvi = new BufferedVectorImage(1, 1);
				ResourceVectorImage source = ResourceVectorImage.loadImage(IMAGE_PATH)) {
			Matrix matrix = new Matrix();

			for (int i = 0; i < ITERATIONS; i++) {
				VectorGraphicsPainter.drawFilteredImage(bvi.getGraphicsContext(), source, matrix, shiftColorMatrix());
			}

			Assert.assertEquals(OUT_OF_MEMORY_FLAGS, bvi.getGraphicsContext().getAndClearDrawingLogFlags());
		}
	}

	/**
	 * Tests that adding too many "draw gradient string" blocks to a BufferedVectorImage causes the proper error flags
	 * to be set.
	 */
	@Test
	public void testDrawGradientString() {

		try (BufferedVectorImage bvi = new BufferedVectorImage(1, 1)) {
			Matrix matrix = new Matrix();
			LinearGradient gradient = new LinearGradient(0f, 0f, 1f, 1f,
					new int[] { Colors.RED, Colors.GREEN, Colors.BLUE });
			final String text = "Sample text";
			VectorFont font = VectorFont.loadFont(FONT_PATH);

			for (int i = 0; i < ITERATIONS; i++) {
				VectorGraphicsPainter.drawGradientString(bvi.getGraphicsContext(), text, font, FONT_SIZE, matrix,
						gradient, ALPHA, BlendMode.SRC_OVER, LETTER_SPACING);
			}

			Assert.assertEquals(OUT_OF_MEMORY_FLAGS, bvi.getGraphicsContext().getAndClearDrawingLogFlags());
		}
	}

	/**
	 * Tests that adding too many "draw gradient string on circle" blocks to a BufferedVectorImage causes the proper
	 * error flags to be set.
	 */
	@Test
	public void testDrawGradientStringOnCircle() {

		try (BufferedVectorImage bvi = new BufferedVectorImage(1, 1)) {
			Matrix matrix = new Matrix();
			LinearGradient gradient = new LinearGradient(0f, 0f, 1f, 1f,
					new int[] { Colors.RED, Colors.GREEN, Colors.BLUE });
			final String text = "Sample text";
			VectorFont font = VectorFont.loadFont(FONT_PATH);

			for (int i = 0; i < ITERATIONS; i++) {
				VectorGraphicsPainter.drawGradientStringOnCircle(bvi.getGraphicsContext(), text, font, FONT_SIZE,
						matrix, gradient, CIRCLE_RADIUS, Direction.CLOCKWISE, ALPHA, BlendMode.SRC_OVER,
						LETTER_SPACING);
			}

			Assert.assertEquals(OUT_OF_MEMORY_FLAGS, bvi.getGraphicsContext().getAndClearDrawingLogFlags());
		}
	}

	/**
	 * Tests that adding too many "draw image" blocks to a BufferedVectorImage causes the proper error flags to be set.
	 */
	@Test
	public void testDrawImage1() {

		try (BufferedVectorImage bvi = new BufferedVectorImage(1, 1);
				ResourceVectorImage source = ResourceVectorImage.loadImage(IMAGE_PATH)) {
			Matrix matrix = new Matrix();

			for (int i = 0; i < ITERATIONS; i++) {
				VectorGraphicsPainter.drawImage(bvi.getGraphicsContext(), source, matrix);
			}

			Assert.assertEquals(OUT_OF_MEMORY_FLAGS, bvi.getGraphicsContext().getAndClearDrawingLogFlags());
		}
	}

	/**
	 * Tests that adding too many "draw image" blocks to a BufferedVectorImage causes the proper error flags to be set.
	 */
	@Test
	public void testDrawImage2() {

		try (BufferedVectorImage bvi = new BufferedVectorImage(1, 1);
				ResourceVectorImage source = ResourceVectorImage.loadImage(IMAGE_PATH)) {
			for (int i = 0; i < ITERATIONS; i++) {
				VectorGraphicsPainter.drawImage(bvi.getGraphicsContext(), source, 0f, 0f);
			}

			Assert.assertEquals(OUT_OF_MEMORY_FLAGS, bvi.getGraphicsContext().getAndClearDrawingLogFlags());
		}
	}

	/**
	 * Tests that adding too many "draw image" blocks to a BufferedVectorImage causes the proper error flags to be set.
	 */
	@Test
	public void testDrawImage3() {

		try (BufferedVectorImage bvi = new BufferedVectorImage(1, 1);
				ResourceVectorImage source = ResourceVectorImage.loadImage(IMAGE_PATH)) {
			Matrix matrix = new Matrix();

			for (int i = 0; i < ITERATIONS; i++) {
				VectorGraphicsPainter.drawImage(bvi.getGraphicsContext(), source, matrix, ALPHA);
			}

			Assert.assertEquals(OUT_OF_MEMORY_FLAGS, bvi.getGraphicsContext().getAndClearDrawingLogFlags());
		}
	}

	/**
	 * Tests that adding too many "draw image" blocks with a BufferedVectorImage as the source to a BufferedVectorImage
	 * causes the proper error flags to be set.
	 */
	@Test
	public void testDrawImage4() {

		try (BufferedVectorImage bvi = new BufferedVectorImage(1, 1);
				BufferedVectorImage source = sourceBufferedVectorImage()) {
			for (int i = 0; i < ITERATIONS; i++) {
				VectorGraphicsPainter.drawImage(bvi.getGraphicsContext(), source, 0f, 0f);
			}

			Assert.assertEquals(OUT_OF_MEMORY_FLAGS, bvi.getGraphicsContext().getAndClearDrawingLogFlags());
		}
	}

	/**
	 * Tests that adding too many "draw image" blocks with an image from the RAM as the source to a BufferedVectorImage
	 * causes the proper error flags to be set.
	 */
	@Test
	public void testDrawImage5() {

		try (BufferedVectorImage bvi = new BufferedVectorImage(1, 1);
				ResourceVectorImage source = ResourceVectorImage.loadImage(IMAGE_PATH);
				ResourceVectorImage filteredSource = source.filterImage(shiftColorMatrix())) {
			for (int i = 0; i < ITERATIONS; i++) {
				VectorGraphicsPainter.drawImage(bvi.getGraphicsContext(), filteredSource, 0f, 0f);
			}

			Assert.assertEquals(OUT_OF_MEMORY_FLAGS, bvi.getGraphicsContext().getAndClearDrawingLogFlags());
		}
	}

	/**
	 * Tests that adding too many "draw string" blocks to a BufferedVectorImage causes the proper error flags to be set.
	 */
	@Test
	public void testDrawString1() {

		try (BufferedVectorImage bvi = new BufferedVectorImage(1, 1)) {
			final String text = "Sample text";
			VectorFont font = VectorFont.loadFont(FONT_PATH);

			for (int i = 0; i < ITERATIONS; i++) {
				VectorGraphicsPainter.drawString(bvi.getGraphicsContext(), text, font, FONT_SIZE, 0f, 0f);
			}

			Assert.assertEquals(OUT_OF_MEMORY_FLAGS, bvi.getGraphicsContext().getAndClearDrawingLogFlags());
		}
	}

	/**
	 * Tests that adding too many "draw string" blocks to a BufferedVectorImage causes the proper error flags to be set.
	 */
	@Test
	public void testDrawString2() {

		try (BufferedVectorImage bvi = new BufferedVectorImage(1, 1)) {
			Matrix matrix = new Matrix();
			final String text = "Sample text";
			VectorFont font = VectorFont.loadFont(FONT_PATH);

			for (int i = 0; i < ITERATIONS; i++) {
				VectorGraphicsPainter.drawString(bvi.getGraphicsContext(), text, font, FONT_SIZE, matrix, ALPHA,
						BlendMode.SRC_OVER, LETTER_SPACING);
			}

			Assert.assertEquals(OUT_OF_MEMORY_FLAGS, bvi.getGraphicsContext().getAndClearDrawingLogFlags());
		}
	}

	/**
	 * Tests that adding too many "draw string on circle" blocks to a BufferedVectorImage causes the proper error flags
	 * to be set.
	 */
	@Test
	public void testDrawStringOnCircle1() {

		try (BufferedVectorImage bvi = new BufferedVectorImage(1, 1)) {
			Matrix matrix = new Matrix();
			final String text = "Sample text";
			VectorFont font = VectorFont.loadFont(FONT_PATH);

			for (int i = 0; i < ITERATIONS; i++) {
				VectorGraphicsPainter.drawStringOnCircle(bvi.getGraphicsContext(), text, font, FONT_SIZE, matrix,
						CIRCLE_RADIUS, Direction.CLOCKWISE, ALPHA, BlendMode.SRC_OVER, LETTER_SPACING);
			}

			Assert.assertEquals(OUT_OF_MEMORY_FLAGS, bvi.getGraphicsContext().getAndClearDrawingLogFlags());
		}
	}

	/**
	 * Tests that adding too many "draw string on circle" blocks to a BufferedVectorImage causes the proper error flags
	 * to be set.
	 */
	@Test
	public void testDrawStringOnCircle2() {

		try (BufferedVectorImage bvi = new BufferedVectorImage(1, 1)) {
			Matrix matrix = new Matrix();
			final String text = "Sample text";
			VectorFont font = VectorFont.loadFont(FONT_PATH);

			for (int i = 0; i < ITERATIONS; i++) {
				VectorGraphicsPainter.drawStringOnCircle(bvi.getGraphicsContext(), text, font, FONT_SIZE, matrix,
						CIRCLE_RADIUS, Direction.CLOCKWISE);
			}

			Assert.assertEquals(OUT_OF_MEMORY_FLAGS, bvi.getGraphicsContext().getAndClearDrawingLogFlags());
		}
	}

	/**
	 * Tests that adding too many "fill gradient path" blocks to a BufferedVectorImage causes the proper error flags to
	 * be set.
	 */
	@Test
	public void testFillGradientPath1() {

		try (BufferedVectorImage bvi = new BufferedVectorImage(1, 1)) {
			Path path = rectanglePath(1f, 1f);
			Matrix matrix = new Matrix();
			LinearGradient gradient = new LinearGradient(0f, 0f, 1f, 1f,
					new int[] { Colors.RED, Colors.GREEN, Colors.BLUE });

			for (int i = 0; i < ITERATIONS; i++) {
				VectorGraphicsPainter.fillGradientPath(bvi.getGraphicsContext(), path, matrix, gradient);
			}

			Assert.assertEquals(OUT_OF_MEMORY_FLAGS, bvi.getGraphicsContext().getAndClearDrawingLogFlags());
		}
	}

	/**
	 * Tests that adding too many "fill gradient path" blocks to a BufferedVectorImage causes the proper error flags to
	 * be set.
	 */
	@Test
	public void testFillGradientPath2() {

		try (BufferedVectorImage bvi = new BufferedVectorImage(1, 1)) {
			Path path = rectanglePath(1f, 1f);
			Matrix matrix = new Matrix();
			LinearGradient gradient = new LinearGradient(0f, 0f, 1f, 1f,
					new int[] { Colors.RED, Colors.GREEN, Colors.BLUE });

			for (int i = 0; i < ITERATIONS; i++) {
				VectorGraphicsPainter.fillGradientPath(bvi.getGraphicsContext(), path, matrix, gradient,
						FillType.WINDING, ALPHA, BlendMode.SRC_OVER);
			}

			Assert.assertEquals(OUT_OF_MEMORY_FLAGS, bvi.getGraphicsContext().getAndClearDrawingLogFlags());
		}
	}

	/**
	 * Tests that adding too many "fill path" blocks to a BufferedVectorImage causes the proper error flags to be set.
	 */
	@Test
	public void testFillPath1() {

		try (BufferedVectorImage bvi = new BufferedVectorImage(1, 1)) {
			Path path = rectanglePath(1f, 1f);
			Matrix matrix = new Matrix();

			for (int i = 0; i < ITERATIONS; i++) {
				VectorGraphicsPainter.fillPath(bvi.getGraphicsContext(), path, matrix);
			}

			Assert.assertEquals(OUT_OF_MEMORY_FLAGS, bvi.getGraphicsContext().getAndClearDrawingLogFlags());
		}
	}

	/**
	 * Tests that adding too many "fill path" blocks to a BufferedVectorImage causes the proper error flags to be set.
	 */
	@Test
	public void testFillPath2() {

		try (BufferedVectorImage bvi = new BufferedVectorImage(1, 1)) {
			Path path = rectanglePath(1f, 1f);

			for (int i = 0; i < ITERATIONS; i++) {
				VectorGraphicsPainter.fillPath(bvi.getGraphicsContext(), path, 0f, 0f);
			}

			Assert.assertEquals(OUT_OF_MEMORY_FLAGS, bvi.getGraphicsContext().getAndClearDrawingLogFlags());
		}
	}

	/**
	 * Tests that adding too many "fill path" blocks to a BufferedVectorImage causes the proper error flags to be set.
	 */
	@Test
	public void testFillPath3() {

		try (BufferedVectorImage bvi = new BufferedVectorImage(1, 1)) {
			Path path = rectanglePath(1f, 1f);
			Matrix matrix = new Matrix();

			for (int i = 0; i < ITERATIONS; i++) {
				VectorGraphicsPainter.fillPath(bvi.getGraphicsContext(), path, matrix, FillType.WINDING, ALPHA,
						BlendMode.SRC_OVER);
			}

			Assert.assertEquals(OUT_OF_MEMORY_FLAGS, bvi.getGraphicsContext().getAndClearDrawingLogFlags());
		}
	}
}
