/*
 * Java
 *
 * Copyright 2019-2021 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.library.iot.rcommand.bluetooth;

import java.util.HashMap;
import java.util.Map;

import ej.annotation.Nullable;
import ej.bluetooth.BluetoothAdapter;
import ej.bluetooth.BluetoothAttribute;
import ej.bluetooth.BluetoothConnection;
import ej.bluetooth.BluetoothService;
import ej.library.iot.rcommand.bluetooth.commands.AddServiceEndpoint;
import ej.library.iot.rcommand.bluetooth.commands.ConnectEndpoint;
import ej.library.iot.rcommand.bluetooth.commands.DisableEndpoint;
import ej.library.iot.rcommand.bluetooth.commands.DisconnectEndpoint;
import ej.library.iot.rcommand.bluetooth.commands.DiscoverServicesEndpoint;
import ej.library.iot.rcommand.bluetooth.commands.EnableEndpoint;
import ej.library.iot.rcommand.bluetooth.commands.SendNotificationEndpoint;
import ej.library.iot.rcommand.bluetooth.commands.SendPairRequestEndpoint;
import ej.library.iot.rcommand.bluetooth.commands.SendPairResponseEndpoint;
import ej.library.iot.rcommand.bluetooth.commands.SendPasskeyResponseEndpoint;
import ej.library.iot.rcommand.bluetooth.commands.SendReadRequestEndpoint;
import ej.library.iot.rcommand.bluetooth.commands.SendReadResponseEndpoint;
import ej.library.iot.rcommand.bluetooth.commands.SendWriteRequestEndpoint;
import ej.library.iot.rcommand.bluetooth.commands.SendWriteResponseEndpoint;
import ej.library.iot.rcommand.bluetooth.commands.StartAdvertisingEndpoint;
import ej.library.iot.rcommand.bluetooth.commands.StartScanningEndpoint;
import ej.library.iot.rcommand.bluetooth.commands.StopAdvertisingEndpoint;
import ej.library.iot.rcommand.bluetooth.commands.StopScanningEndpoint;
import ej.library.iot.rcommand.bluetooth.notifications.EventNotification;
import ej.rcommand.RemoteCommandManager;
import ej.rcommand.RemoteNotification;
import ej.rcommand.synchronous.Endpoint;
import ej.rcommand.synchronous.EndpointListener;

public class BluetoothController {

	private static final BluetoothController INSTANCE = new BluetoothController();

	private final EventSerializer eventSerializer;

	private final BluetoothDatabase database;
	private final Map<BluetoothConnection, Short> connections;
	private final Map<BluetoothConnection, BluetoothDatabase> connectionDatabases;

	private short connectionHandleCounter;
	private short attributeHandleCounter;

	private @Nullable RemoteCommandManager rcommandManager;

	private BluetoothController() {
		this.eventSerializer = new EventSerializer(this);

		this.database = new BluetoothDatabase();
		this.connections = new HashMap<>();
		this.connectionDatabases = new HashMap<>();

		this.connectionHandleCounter = 1;
		this.attributeHandleCounter = 1;
	}

	public static BluetoothController getInstance() {
		return INSTANCE;
	}

	public void setup(RemoteCommandManager rcommandManager) {
		Endpoint[] endpoints = new Endpoint[] { //
				new EnableEndpoint(this), //
				new DisableEndpoint(this), //
				new StartScanningEndpoint(), //
				new StopScanningEndpoint(), //
				new StartAdvertisingEndpoint(), //
				new StopAdvertisingEndpoint(), //
				new ConnectEndpoint(), //
				new DisconnectEndpoint(this), //
				new SendPairRequestEndpoint(this), //
				new SendPairResponseEndpoint(this), //
				new SendPasskeyResponseEndpoint(this), //
				new DiscoverServicesEndpoint(this), //
				new AddServiceEndpoint(this), //
				new SendReadRequestEndpoint(this), //
				new SendWriteRequestEndpoint(this), //
				new SendReadResponseEndpoint(this), //
				new SendWriteResponseEndpoint(this), //
				new SendNotificationEndpoint(this) //
		};

		for (Endpoint endpoint : endpoints) {
			assert (endpoint != null);
			rcommandManager.registerListener(new EndpointListener(endpoint));
		}

		this.rcommandManager = rcommandManager;
	}

	public void sendEvent(byte[] event) {
		RemoteCommandManager rcommandManager = this.rcommandManager;
		if (rcommandManager != null) {
			RemoteNotification notification = new EventNotification(event, event.length);
			rcommandManager.sendNotification(notification);
		}
	}

	public @Nullable BluetoothConnection getConnection(short connHandle) {
		synchronized (this) {
			for (Map.Entry<BluetoothConnection, Short> entry : this.connections.entrySet()) {
				Short handle = entry.getValue();
				assert (handle != null);
				if (handle.shortValue() == connHandle) {
					return entry.getKey();
				}
			}
			return null;
		}
	}

	public @Nullable Short getConnHandle(BluetoothConnection connection) {
		synchronized (this) {
			return this.connections.get(connection);
		}
	}

	public @Nullable BluetoothAttribute getLocalAttribute(short attributeHandle) {
		synchronized (this) {
			return this.database.getAttribute(attributeHandle);
		}
	}

	public @Nullable Short getLocalAttributeHandle(BluetoothAttribute attribute) {
		synchronized (this) {
			return this.database.getAttributeHandle(attribute);
		}
	}

	public @Nullable BluetoothAttribute getRemoteAttribute(@Nullable BluetoothConnection connection,
			short attributeHandle) {
		if (connection == null) {
			return null;
		}

		synchronized (this) {
			BluetoothDatabase database = this.connectionDatabases.get(connection);
			if (database != null) {
				return database.getAttribute(attributeHandle);
			} else {
				return null;
			}
		}
	}

	public @Nullable Short getRemoteAttributeHandle(BluetoothConnection connection, BluetoothAttribute attribute) {
		synchronized (this) {
			BluetoothDatabase database = this.connectionDatabases.get(connection);
			if (database != null) {
				return database.getAttributeHandle(attribute);
			} else {
				return null;
			}
		}
	}

	public void onAdapterEnabled() {
		BluetoothAdapter.getAdapter().setConnectionListener(this.eventSerializer);
	}

	public void onAdapterDisabled() {
		synchronized (this) {
			this.database.clear();
			this.connections.clear();
			this.connectionDatabases.clear();
		}
	}

	public short onConnected(BluetoothConnection connection) {
		synchronized (this) {
			short connHandle = this.connectionHandleCounter;
			this.connectionHandleCounter++;
			this.connections.put(connection, Short.valueOf(connHandle));
			this.connectionDatabases.put(connection, new BluetoothDatabase());
			return connHandle;
		}
	}

	public @Nullable Short onDisconnected(BluetoothConnection connection) {
		synchronized (this) {
			this.connectionDatabases.remove(connection);
			return this.connections.remove(connection);
		}
	}

	public void onServiceAdded(BluetoothService service, short[] handles) {
		service.setLocalListener(this.eventSerializer);

		synchronized (this) {
			this.attributeHandleCounter = this.database.addService(service, this.attributeHandleCounter, handles);
		}
	}

	public @Nullable Short onServiceDiscovered(BluetoothConnection connection, BluetoothService service) {
		service.setRemoteListener(this.eventSerializer);

		synchronized (this) {
			BluetoothDatabase database = this.connectionDatabases.get(connection);
			if (database != null) {
				Short identicalServiceHandle = database.getServiceHandle(service);
				if (identicalServiceHandle == null) {
					this.attributeHandleCounter = database.addService(service, this.attributeHandleCounter, null);
					return Short.valueOf(this.attributeHandleCounter);
				} else {
					return identicalServiceHandle;
				}
			}
		}

		return null;
	}
}
