/*
 * Java
 *
 * Copyright 2015-2024 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.kf.util.service;

import ej.annotation.NonNull;
import ej.annotation.Nullable;
import ej.basictool.map.AbstractPackedMap;
import ej.basictool.map.PackedMap;
import ej.kf.Feature;
import ej.kf.Feature.State;
import ej.kf.FeatureStateListener;
import ej.kf.Kernel;
import ej.kf.Module;
import ej.kf.Proxy;
import ej.service.ServicePermission;
import ej.service.registry.SimpleServiceRegistry;


/**
 * Implementation of the shared service registry over Kernel and Features.
 * <p>
 * This service registry is able to handle the following kind of services:
 * <ul>
 * <li>Services owned by the kernel, available to the kernel and the features (provided that the type is part of the
 * API)</li>
 * <li>Services owned by a feature using shared interfaces mechanism, available to other features through {@link Proxy}
 * instances</li>
 * <li>Services owned by a feature using an interface owned by the kernel, only available to the kernel (kernel
 * extensions)</li>
 * <li>Services owned by a feature using a type owned by the feature, only available to the feature</li>
 * </ul>
 */
public class SharedServiceRegistryKF extends SimpleServiceRegistry
implements SharedServiceRegistry, FeatureStateListener {
	/**
	 * Map which contains shared services registered by features.
	 *
	 * Methods modifying these maps are synchronized using the monitor of the map.
	 */
	private final PackedMap<Feature, ClassPackedMap> sharedServicesByFeatures;

	/**
	 * Creates a shared service registry.
	 */
	public SharedServiceRegistryKF() {
		this.sharedServicesByFeatures = new PackedMap<>();
	}

	@Override
	public <T> void register(Class<T> serviceClass, T implementation) {
		@SuppressWarnings("null")
		@NonNull
		Module serviceOwner = Kernel.getOwner(implementation);
		@SuppressWarnings("null")
		@NonNull
		Module contextOwner = Kernel.getContextOwner();
		if (contextOwner != serviceOwner) {
			throw new IllegalStateException();
		}
		if (serviceOwner == Kernel.getInstance()) {
			// Service owned by the Kernel.
			super.register(serviceClass, implementation);

		} else {
			// Service owned by a Feature.
			checkServicePermission(serviceClass, ServicePermission.REGISTER_ACTION);

			Kernel.enter();

			ClassPackedMap sharedServices;
			PackedMap<Feature, ClassPackedMap> sharedServicesByFeatures = this.sharedServicesByFeatures;

			synchronized (sharedServicesByFeatures) {
				if (sharedServicesByFeatures.isEmpty()) {
					Kernel.addFeatureStateListener(this);
				}

				Feature featureOwner = (Feature) serviceOwner;
				sharedServices = sharedServicesByFeatures.get(featureOwner);
				if (sharedServices == null) {
					// This is the first time this Feature registers a service.
					sharedServices = new ClassPackedMap();
					sharedServicesByFeatures.put(featureOwner, sharedServices);
				}
			}

			synchronized (sharedServices) {
				sharedServices.put(serviceClass, implementation);
			}
		}
	}


	@Override
	public <T> void unregister(Class<T> serviceClass, T implementation) {
		@SuppressWarnings("null")
		@NonNull
		Module serviceOwner = Kernel.getOwner(implementation);
		@SuppressWarnings("null")
		@NonNull
		Module contextOwner = Kernel.getContextOwner();
		if (contextOwner != serviceOwner) {
			throw new IllegalStateException();
		}
		if (serviceOwner == Kernel.getInstance()) {
			// Service owned by the Kernel.
			super.unregister(serviceClass, implementation);
		} else {
			// Service owned by a Feature.
			checkServicePermission(serviceClass, ServicePermission.UNREGISTER_ACTION);

			Kernel.enter();

			PackedMap<Feature, ClassPackedMap> sharedServicesByFeatures = this.sharedServicesByFeatures;

			ClassPackedMap sharedServices = sharedServicesByFeatures.get(contextOwner);
			if (sharedServices != null) {
				synchronized (sharedServices) {
					if (sharedServices.get(serviceClass) == implementation) {
						sharedServices.remove(serviceClass);
						if (sharedServices.isEmpty()) {
							// This Feature no longer register some services.
							synchronized (sharedServicesByFeatures) {
								sharedServicesByFeatures.remove(contextOwner);
								if (sharedServicesByFeatures.isEmpty()) {
									Kernel.removeFeatureStateListener(this);
								}
							}
						}
					}
				}
			}
		}
	}

	/**
	 * Gets the registered implementation for the given service.
	 *
	 * This implementation depends on who request it. if it is a feature we search the service with the following order:
	 * <ul>
	 * <li>We search it in the shared services provided by the current feature.</li>
	 * <li>We search it in the shared services provided by the kernel.</li>
	 * <li>We search it in the shared services provided by the trusted features.</li>
	 * <li>We search it in the shared services provided by the non trusted features.</li>
	 * </ul>
	 * if it is the kernel, we search the service with the following order:
	 * <ul>
	 * <li>We search it in the shared services provided by the kernel.</li>
	 * <li>We search it in the shared services provided by the trusted features.</li>
	 * <li>We search it in the shared services provided by the non trusted features.</li>
	 * </ul>
	 *
	 *
	 * @param serviceClass
	 *            the service to found.
	 * @param <T>
	 *            the service type.
	 * @return an instance of the given service or {@code null} if no implementation is available.
	 */
	@Override
	@Nullable
	public <T> T getService(Class<T> serviceClass) {
		checkServicePermission(serviceClass, ServicePermission.GET_ACTION);

		// Search the service in the ones owned by the current context owner.
		@SuppressWarnings("null")
		@NonNull
		Module contextOwner = Kernel.getContextOwner();
		if (!Kernel.isInKernelMode()) {
			ClassPackedMap sharedServices = this.sharedServicesByFeatures.get(contextOwner);

			if (sharedServices != null) {
				T implementation = getSharedService(contextOwner, serviceClass, sharedServices);
				if (implementation != null) {
					return implementation;
				}
			}
		}

		// Search the service in the ones provided by the kernel.
		T kernelService = super.getService(serviceClass);
		if (kernelService != null) {
			return kernelService;
		}

		Kernel.enter();

		// Search the service in the ones provided by features.
		for (ClassPackedMap sharedServices : this.sharedServicesByFeatures.values()) {
			T implementation = getSharedService(contextOwner, serviceClass, sharedServices);
			if (implementation != null) {
				return implementation;
			}
		}

		return null;
	}

	@Nullable
	private <T> T getSharedService(Module contextOwner, Class<T> serviceClass, ClassPackedMap sharedServices) {
		@SuppressWarnings("unchecked")
		T implementation = (T) sharedServices.get(serviceClass);
		if (implementation != null) {
			if ((contextOwner == Kernel.getInstance() && serviceClass.isAssignableFrom(implementation.getClass()))
					|| contextOwner == Kernel.getOwner(implementation)) {
				// Kernel context and service is an implementation of a Kernel
				// API class
				// or Feature context and Feature implementation
				// => implementation is directly accessible to the caller
				return implementation;
			} else if (Kernel.isSharedInterface(serviceClass)) {
				// implementation of a shared interface accessible to an
				// other Feature through a Proxy.

				try {
					return Kernel.bind(implementation, serviceClass, (Feature) Kernel.getOwner(serviceClass));
				} catch (IllegalAccessError e) {
					// Ignore the case when an Interface with a same name is registered in the
					// sharedRegistry but is **not** declared as sharedInterface
				}
			}
		}

		return null;
	}

	@Override
	public void stateChanged(@Nullable Feature feature, @Nullable State previousState) {
		if (feature != null && feature.getState() == State.STOPPED) {
			// unregister services that would have been registered by the
			// Feature being removed
			synchronized (this.sharedServicesByFeatures) {
				this.sharedServicesByFeatures.remove(feature);
			}
		}
	}

	class ClassPackedMap extends AbstractPackedMap<Class<?>, Object> {

		public ClassPackedMap() {
			super();
		}

		public ClassPackedMap(AbstractPackedMap<Class<?>, Object> map) {
			super(map);
		}

		@Override
		public Object clone() {
			return new ClassPackedMap(this);
		}

		@Override
		protected boolean isSame(@Nullable Object key, @Nullable Object candidateKey) {
			if (key != null && candidateKey != null) {
				Class<?> keyClass = (Class<?>) key;
				Class<?> candidateKeyClass = (Class<?>) candidateKey;
				String name = keyClass.getName();
				try {
					name = Kernel.clone(name, Kernel.getInstance());
				} catch (CloneNotSupportedException e) {
					throw new AssertionError(e);
				}
				return name.equals(candidateKeyClass.getName());
			}
			return false;
		}

		@Override
		protected int getKeyHashCode(@Nullable Object wrappedKey) {
			if (wrappedKey != null) {
				String name = ((Class<?>) wrappedKey).getName();
				int hashCode = name.hashCode();
				return hashCode;
			}
			return 0;
		}

	}

}
