/*
 * Java
 *
 * Copyright 2012-2022 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.net.test.util;

import java.io.IOException;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.SocketException;
import java.util.Enumeration;

import ej.ecom.network.IPConfiguration;
import ej.ecom.network.NetworkInterfaceManager;
import ej.ecom.wifi.AccessPoint;
import ej.ecom.wifi.WifiManager;

public class WifiNetInit implements INetInit {

	private static final String WIFI_PROP_PREFIX = "wifi.";

	private static final String WIFI_PROP_PASSPHRASE = WIFI_PROP_PREFIX+"passphrase";
	private static final String WIFI_PROP_SSID = WIFI_PROP_PREFIX+"ssid";

	private static final String WIFI_PROP_JOIN_RETRY = WIFI_PROP_PREFIX+"join.nbretry";
	private static final int WIFI_PROP_JOIN_RETRY_VALUE_DEFAULT = 3;
	private static final Integer WIFI_PROP_JOIN_RETRY_VALUE = Integer.getInteger(WIFI_PROP_JOIN_RETRY, WIFI_PROP_JOIN_RETRY_VALUE_DEFAULT);

	// IPv4 address property values:
	// ip=dhcp == IPv4 address provided by DHCP
	// ip=xx.xx.xx.xx == IPv4 address provided by the user
	private static final String NETIF_CONF_PROP_IP = WIFI_PROP_PREFIX+"ip";
	public static final String DEFAULT_NETIF_CONF_PROP_IP_VALUE = "dhcp";

	public static final String NETIF_IPV4_NAME_PROPERTY = "netif.ipv4.name";
	public static final String NETIF_IPV6_NAME_PROPERTY = "netif.ipv6.name";
	public static final String NETIF_NAME_PROPERTY = "netif.name";
	public static final String NETIF_NAME_VALUE_DEFAULT = "wifi_interface";
	public static final String PREFER_IPV6_PROPERTY = "testsuite.preferipv6";

	private NetworkInterface netif;

	@Override
	public void initialize() throws IOException {
		String passphrase = System.getProperty(WIFI_PROP_PASSPHRASE);
		String ssid = System.getProperty(WIFI_PROP_SSID);

		// sanity check on wifi settings
		if ((passphrase != null) && !passphrase.equals("") && (ssid != null) && (!ssid.equals(""))) {
			// start the Wifi network interface and join the required
			// access point

			startWifiInterface();
			WifiManager wifiManager = WifiManager.getInstance();
			int nbTries = WIFI_PROP_JOIN_RETRY_VALUE;
			while(--nbTries >= 0) {
				try{
					log("Trying to join access point \""+ssid+"\"...");
					wifiManager.join(ssid, passphrase);
					break;
				} catch(IOException e){
					log("Join failed");
					if(nbTries == 0){
						// access point cannot be joined, all tries have failed
						throw e;
					}
					wifiManager.leave();
				}
			}
			logWifiState(ssid, wifiManager);
		} else {
			log("Wifi settings not set (ssid=" + ssid + ", passphrase=" + passphrase + ")");
		}
	}

	private static NetworkInterface FoundInterface(String interfaceName) throws IOException {
		Enumeration<NetworkInterface> interfaces = NetworkInterface.getNetworkInterfaces();

		/**
		 * Check if we have a valid enumeration of network interfaces
		 */
		if (interfaces == null) {
			throw new IOException("No network interfaces found");
		}

		/**
		 * Look through all of the enumeration and compare with the one to be
		 * found
		 */
		while (interfaces.hasMoreElements()) {
			NetworkInterface networkInterface = interfaces.nextElement();

			if (networkInterface.getName().compareTo(interfaceName) == 0) {
				return networkInterface;
			}
		}

		return null;
	}

	private String getInterfaceName() {
		if (System.getProperty(NETIF_NAME_PROPERTY) != null) {
			return System.getProperty(NETIF_NAME_PROPERTY);
		}
		if (System.getProperty(PREFER_IPV6_PROPERTY).equals("true")) {
			return System.getProperty(NETIF_IPV6_NAME_PROPERTY, NETIF_NAME_VALUE_DEFAULT);
		} else {
			return System.getProperty(NETIF_IPV4_NAME_PROPERTY, NETIF_NAME_VALUE_DEFAULT);
		}
	}

	private void startWifiInterface() throws SocketException, IllegalStateException, IOException {
		// select the first NetworkInterface and start it
		// predicate: the first network interface found is a Wifi network
		// interface
		String interfaceName = getInterfaceName();
		netif = FoundInterface(interfaceName);
		if (netif == null) {
			throw new IOException("No Wifi interface \"" + interfaceName + "\" found");
		}
		if (!NetworkInterfaceManager.isStarted(netif)) {
			NetworkInterfaceManager.start(netif);
		}
		// force to enable network interface
		if (!NetworkInterfaceManager.isEnabled(netif)) {
			NetworkInterfaceManager.enable(netif);
		}


		// Set IP configuration for Wifi module
		IPConfiguration conf = new IPConfiguration();
		// retrieve the network interface settings
		String ipConf = System.getProperty(NETIF_CONF_PROP_IP, DEFAULT_NETIF_CONF_PROP_IP_VALUE);
		if (ipConf.equals(DEFAULT_NETIF_CONF_PROP_IP_VALUE)) {
			conf.useDHCP(true);
		} else {
			conf.useDHCP(false);
			conf.setIP(InetAddress.getByName(ipConf));
		}
		try {
			NetworkInterfaceManager.configure(netif, conf);
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	private void logWifiState(String ssid, WifiManager wifiManager) throws IOException {
		// log Network interface status
		IPConfiguration wifiIPConf = NetworkInterfaceManager.getIPConfiguration(netif);
		StringBuilder sb = new StringBuilder();
		sb.append("NetIF update [ip=");
		sb.append(wifiIPConf.getIP());
		sb.append(", netmask=");
		sb.append(wifiIPConf.getNetmask());
		sb.append(", gateway=");
		sb.append(wifiIPConf.getGateway());
		InetAddress[] dnsAddrs = wifiIPConf.getDNS();
		int nbDNS = dnsAddrs.length;
		if ((dnsAddrs != null) && (nbDNS > 0)) {
			sb.append(", dns=");
			if (nbDNS == 1) {
				sb.append(dnsAddrs[0]);
			} else {
				sb.append("(");
				for (int i = -1; ++i < nbDNS - 1;) {
					sb.append(dnsAddrs[i]);
					sb.append(",");
				}
				sb.append(dnsAddrs[nbDNS - 1]);
				sb.append(")");
			}
		}
		sb.append("]");
		log(sb.toString());

		// log Wifi connection status
		AccessPoint ap = wifiManager.getJoined();
		log("Wifi AP joined " + ap);
	}

	private void log(String msg) {
		System.out.println("[INFO] "+msg);
	}

}