/*
 * Copyright 2015-2021 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.storage;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import ej.annotation.Nullable;

/**
 * Defines the storage mechanism.
 * <p>
 * The storage uses an identifier to store, retrieve or remove an entry. An identifier is an unlimited-length sequence
 * (greater than 0) of letters, digits and special characters, the first of which must be a letter. The "letters"
 * include lowercase(a-z) and uppercase(A-Z) ASCII latin characters and ASCII underscore(_). The "digits" include the
 * ASCII digits(0-9). The special characters include ASCII hyphen(-) and ASCII dot(.).
 * <p>
 * This API does not ensure that concurrent accesses on the same ID are thread-safe.
 *
 */
public interface Storage {

	/**
	 * Stores data that will be referenced with an ID.
	 * <p>
	 * If an entry is already stored with this ID, its existing data is fully removed and replaced by the given data.
	 * Otherwise, a new entry is created.
	 * <p>
	 * This operation may block the current thread depending on the implementation.
	 * <p>
	 * The returned output stream must be flushed or closed to ensure that the data is available. Example of usage:
	 *
	 * <pre>
	 * try(OutputStream outputStream = store(id)) {
	 * 	outputStream.write(…);
	 * 	outputStream.write(…);
	 * }
	 * </pre>
	 *
	 * @param id
	 *            the ID of the stored entry.
	 * @return the output stream to write data.
	 * @throws IOException
	 *             if an I/O error occurs.
	 * @throws IllegalArgumentException
	 *             if the given ID is not a valid identifier.
	 */
	OutputStream store(String id) throws IOException;

	/**
	 * Modifies the entry that is referenced by an ID.
	 * <p>
	 * If an entry is already stored with this ID, its existing data is not removed and only the targeted bytes will be
	 * overwritten. Otherwise, a new entry is created.
	 * <p>
	 * This operation may block the current thread depending on the implementation.
	 * <p>
	 * The returned output stream must be flushed or closed to ensure that the data is available.
	 *
	 * @param id
	 *            the ID of the stored entry.
	 * @param offset
	 *            the offset at which to write data.
	 * @return the output stream to write data.
	 * @throws IOException
	 *             if an I/O error occurs.
	 * @throws IllegalArgumentException
	 *             if the given ID is not a valid identifier or if the given offset is negative.
	 */
	OutputStream modify(String id, int offset) throws IOException;

	/**
	 * Appends some data to an entry referenced with an ID.
	 * <p>
	 * If an entry is already stored with this ID, its existing data is not removed and the given data will be added to
	 * the existing entry. Otherwise, a new entry is created.
	 * <p>
	 * This operation may block the current thread depending on the implementation.
	 * <p>
	 * The returned output stream must be flushed or closed to ensure that the data is available. Example of usage:
	 *
	 * <pre>
	 * try(OutputStream outputStream = append(id)) {
	 * 	outputStream.write(…);
	 * 	outputStream.write(…);
	 * }
	 * </pre>
	 *
	 * @param id
	 *            the ID of the stored entry.
	 * @return the output stream to write data.
	 * @throws IOException
	 *             if an I/O error occurs.
	 * @throws IllegalArgumentException
	 *             if the given ID is not a valid identifier.
	 */
	OutputStream append(String id) throws IOException;

	/**
	 * Loads the data stored with a specific ID or {@code null} if there is none.
	 * <p>
	 * The returned input stream must be closed by the caller.
	 * <p>
	 * This operation may block the current thread depending on the implementation.
	 *
	 * @param id
	 *            the ID of the entry to be returned.
	 * @return the data of the entry stored with the given ID or {@code null}.
	 * @throws IOException
	 *             if an I/O error occurs.
	 * @throws IllegalArgumentException
	 *             if the given ID is not a valid identifier.
	 */
	@Nullable
	InputStream load(String id) throws IOException;

	/**
	 * Changes the ID of an entry.
	 * <p>
	 * If an entry is already stored with the destination ID, it is overwritten.
	 * <p>
	 * This operation may block the current thread depending on the implementation.
	 *
	 * @param sourceId
	 *            the old ID of the entry.
	 * @param destinationId
	 *            the new ID of the entry.
	 * @throws IOException
	 *             if an I/O error occurs.
	 * @throws IOException
	 *             if there is no entry stored with the source ID.
	 * @throws IllegalArgumentException
	 *             if one of the given IDs is not a valid identifier.
	 */
	void move(String sourceId, String destinationId) throws IOException;

	/**
	 * Removes the entry stored with an ID.
	 * <p>
	 * This operation may block the current thread depending on the implementation.
	 *
	 * @param id
	 *            the ID of the entry to remove.
	 * @throws IOException
	 *             if an I/O error occurs.
	 * @throws IOException
	 *             if there is no entry stored with the given ID.
	 * @throws IllegalArgumentException
	 *             if the given ID is not a valid identifier.
	 */
	void remove(String id) throws IOException;

	/**
	 * Returns the size of the entry stored with an ID.
	 *
	 * @param id
	 *            the ID of the entry.
	 * @throws IOException
	 *             if an I/O error occurs.
	 * @throws IOException
	 *             if there is no entry stored with the given ID.
	 * @throws IllegalArgumentException
	 *             if the given ID is not a valid identifier.
	 * @return the size of the entry.
	 */
	long getSize(String id) throws IOException;

	/**
	 * Tests whether an entry exists for this ID.
	 *
	 * @param id
	 *            the ID to check.
	 * @throws IOException
	 *             if an I/O error occurs.
	 * @throws IllegalArgumentException
	 *             if the given ID is not a valid identifier.
	 * @return <code>true</code> if the given ID exists, <code>false</code> otherwise
	 */
	boolean exists(String id) throws IOException;

	/**
	 * Returns all IDs of the stored entries.
	 *
	 * @return an array of all IDs.
	 * @throws IOException
	 *             if an I/O error occurs.
	 */
	String[] getIds() throws IOException;
}
