/*
 * Java
 *
 * Copyright 2024 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microui.display;

import ej.fp.Image;

/**
 * Represents a MicroUI Image.
 * <p>
 * This interface is used by several drawing methods which use an image as source image. It can be mapped on byte array
 * given as parameter in some MicroUI natives. This byte array is retrieved in MicroEJ application using the method
 * Image.getSNIContext().
 *
 * @see LLUIDisplay#mapMicroUIImage(byte[])
 * @see LLUIDisplay#newMicroUIImage(byte[])
 */
public interface MicroUIImage {

	/**
	 * Returns the MicroUI Image width.
	 *
	 * @return image width in pixels.
	 */
	int getWidth();

	/**
	 * Returns the MicroUI Image height.
	 *
	 * @return image height in pixels.
	 */
	int getHeight();

	/**
	 * Returns the MicroUI Image format.
	 *
	 * @return image format.
	 */
	MicroUIImageFormat getFormat();

	/**
	 * Returns the front panel image allocated for the MicroUI Image.
	 *
	 * @return the MicroUI Image representation in front panel framework.
	 */
	Image getImage();

	/**
	 * Returns the front panel image allocated for the MicroUI Image, applying a rendering color.
	 *
	 * This color is only useful when the MicroUI Image format is {@link MicroUIImageFormat#MICROUI_IMAGE_FORMAT_A8},
	 * {@link MicroUIImageFormat#MICROUI_IMAGE_FORMAT_A4}, {@link MicroUIImageFormat#MICROUI_IMAGE_FORMAT_A2} or
	 * {@link MicroUIImageFormat#MICROUI_IMAGE_FORMAT_A1}.
	 *
	 * Next calls to {@link #getImage()} (in same native function) will return the same image with same rendering color
	 * until next call to {@link #getImage(int)}.
	 *
	 * For all other formats, the color is ignored and this call is equivalent than calling {@link #getImage()}.
	 *
	 * @param color
	 *            the color to apply on Ax image in 0xRRGGBB format (the alpha component is ignored).
	 *
	 * @return the MicroUI Image representation in front panel framework.
	 */
	Image getImage(int color);

	/**
	 * Returns true when the image targets the LCD.
	 *
	 * @return true when the image targets the LCD.
	 */
	boolean isLCD();

	/**
	 * Returns true when the image has been closed by the MicroEJ application.
	 *
	 * @return true when the image has been closed.
	 */
	boolean isClosed();

	/**
	 * Returns the 32-bit ARGB color format (A-R-G-B) of a pixel.
	 *
	 * @param x
	 *            the x coordinate of the pixel.
	 * @param y
	 *            the y coordinate of the pixel.
	 *
	 * @return an ARGB8888 color or 0 if the pixel is out-of-bounds (same behavior as embedded side).
	 */
	int readPixel(int x, int y);

	/**
	 * Requests a reading on the graphics context.
	 *
	 * @return true if reading can be performed, false when the reading does not need to be done for any reason.
	 */
	boolean requestReading();
}
