/*
 * Java
 *
 * Copyright 2008-2020 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microui.led;

import ej.fp.Widget;

/**
 * This interface holds all methods required by MicroUI implementation to be compatible with the MicroUI LEDs engine.
 * <p>
 * An implementation of this interface is required to run a MicroUI application which uses the class MicroUI Led;
 * otherwise some errors will be thrown at runtime and the application will not able to run correctly. This
 * implementation should be a widget which at least allows to show an image when the LED is off and another image when
 * the LED is on.
 * <p>
 * The LED engine retrieves on startup all widgets which implement this interface (using method
 * <code>Device.getDevice().getWidgets(LLUILedImpl.class)</code>). Then it considers it can target a LED using the
 * widget label ({@link Widget#getLabel()}). A widget label must be an integer; it allows to retrieve a LED from an
 * integer index. The first LED index must be <code>0</code> and last LED index must be
 * <code>Device.getDevice().getWidgets(LLUILedImpl.class).size() - 1</code>.
 * <ul>
 * <li>If an index in the range does not target a valid widget, the LED action (on/off) is not performed but no error is
 * thrown.</li>
 * <li>If a LED has an "integer" label smaller than 0, this LED will be never used.</li>
 * <li>If a LED has an "integer" label higher or equals to
 * <code>Device.getDevice().getWidgets(LLUILedImpl.class).size()</code>, this LED will be never used.</li>
 * <li>If a LED has a "non-interger" label this LED will be never used.</li>
 * </ul>
 *
 */
public interface LLUILedImpl {

	/**
	 * Minimal value of intensity according MicroUI specification.
	 */
	int MIN_INTENSITY = 0;

	/**
	 * Maximal value of intensity according MicroUI specification.
	 */
	int MAX_INTENSITY = 255;

	/**
	 * Sets the intensity of this led.
	 *
	 * @param intensity
	 *            the intensity to set.
	 */
	void setIntensity(int intensity);

	/**
	 * Gets the intensity of this led.
	 *
	 * @return intensity the led intensity.
	 */
	int getIntensity();

}
