/*
 * Java
 *
 * Copyright 2009-2020 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.fp.widget;

import java.util.concurrent.atomic.AtomicInteger;

import ej.fp.FrontPanel;
import ej.fp.Image;
import ej.fp.Widget;
import ej.fp.Widget.WidgetAttribute;
import ej.fp.Widget.WidgetDescription;
import ej.microui.display.LLUIDisplayImpl;
import ej.microui.display.LLUIPainter.MicroUIGraphicsContext;
import ej.microui.display.LLUIPainter.MicroUIImageFormat;

/**
 * This widget is an implementation of MicroUI {@link LLUIDisplayImpl} interface. An implementation of this interface is
 * required to be able to use a MicroUI Display in a MicroEJ application.
 * <p>
 * This widget has been implemented to target a standard display: a display whose pixel format is
 * {@link MicroUIImageFormat#MICROUI_IMAGE_FORMAT_ARGB8888} , {@link MicroUIImageFormat#MICROUI_IMAGE_FORMAT_RGB888},
 * {@link MicroUIImageFormat#MICROUI_IMAGE_FORMAT_RGB565}, {@link MicroUIImageFormat#MICROUI_IMAGE_FORMAT_ARGB1555},
 * {@link MicroUIImageFormat#MICROUI_IMAGE_FORMAT_ARGB4444}, {@link MicroUIImageFormat#MICROUI_IMAGE_FORMAT_C4},
 * {@link MicroUIImageFormat#MICROUI_IMAGE_FORMAT_C2} or {@link MicroUIImageFormat#MICROUI_IMAGE_FORMAT_C1}. To avoid to
 * create a subclass of this class when the pixel format is custom, an extension of this widget can be set in front
 * panel description file (fp file) using the widget attribute <code>extensionClass</code>.
 * <p>
 * This widget manages the double buffering mode (see {@link LLUIDisplayImpl#isDoubleBuffered()}. However all involved
 * methods by this notion are implemented according the value returned by {@link #isDoubleBuffered()}. The optional
 * widget attribute <code>doubleBufferFeature</code> can be configured to override default behavior (enabled by
 * default).
 * <p>
 * This widget manages a backlight with dimming (0 to 100%) (see {@link LLUIDisplayImpl#hasBacklight()}. However all
 * involved methods by this notion are implemented according the value returned by {@link #hasBacklight()}. The optional
 * widget attribute <code>backlightFeature</code> can be configured to override default behavior (enabled by default).
 * <p>
 * The filter image is used to define the display rendering area. Outside this area, the display buffer data is not
 * drawn. This image must have the same size (@see {@link #setWidth(int)} and {@link #setHeight(int)}) than the widget
 * itself. If not, an error is thrown in {@link #finalizeConfiguration()}.
 */
@SuppressWarnings("nls")
@WidgetDescription(attributes = { @WidgetAttribute(name = "label", isOptional = true), @WidgetAttribute(name = "x"),
		@WidgetAttribute(name = "y"), @WidgetAttribute(name = "width"), @WidgetAttribute(name = "height"),
		@WidgetAttribute(name = "displayWidth", isOptional = true),
		@WidgetAttribute(name = "displayHeight", isOptional = true),
		@WidgetAttribute(name = "initialColor", isOptional = true), @WidgetAttribute(name = "alpha", isOptional = true),
		@WidgetAttribute(name = "doubleBufferFeature", isOptional = true),
		@WidgetAttribute(name = "backlightFeature", isOptional = true),
		@WidgetAttribute(name = "filter", isOptional = true),
		@WidgetAttribute(name = "extensionClass", isOptional = true) })
public class Display extends Widget implements LLUIDisplayImpl {

	/**
	 * This interface is a subset of {@link LLUIDisplayImpl} interface (provided by MicroUI graphical engine and
	 * required to use a MicroUI Display in a MicroEJ application).
	 * <p>
	 * It provides a set of minimal methods to customize the {@link Display} widget (which implements
	 * {@link LLUIDisplayImpl}) without needing to subclass it (no need to create a sub-widget "display" with same
	 * characteritics and same {@link WidgetAttribute}, just to override one setting).
	 * <p>
	 * An implementation classname of this interface can be set thanks the widget attribute
	 * {@link Display#setExtensionClass(String)}.
	 */
	public static interface DisplayExtension {

		/**
		 * Asks if the display is a colored display or not.
		 *
		 * @param display
		 *            the display widget.
		 *
		 * @return true when the display is not a grayscale display, false otherwise.
		 * @see LLUIDisplayImpl#isColor()
		 */
		boolean isColor(Display display);

		/**
		 * Gets the number of colors that can be represented on the device.
		 *
		 * @param display
		 *            the display widget.
		 *
		 * @return the number of colors of the display.
		 * @see LLUIDisplayImpl#getNumberOfColors()
		 */
		int getNumberOfColors(Display display);

		/**
		 * Converts the 32-bit ARGB color format (A-R-G-B) into the display color format.
		 *
		 * @param display
		 *            the display widget.
		 * @param argbColor
		 *            the color to convert.
		 * @return the display color equivalent to <code>microUIColor</code>.
		 * @see LLUIDisplayImpl#convertARGBColorToDisplayColor(int)
		 */
		int convertARGBColorToDisplayColor(Display display, int argbColor);

		/**
		 * Converts the display color format into a 32-bit ARGB color format (A-R-G-B).
		 *
		 * @param display
		 *            the display widget.
		 * @param displayColor
		 *            the color to convert.
		 * @return the MicroUI color equivalent to <code>displayColor</code>.
		 * @see LLUIDisplayImpl#convertDisplayColorToARGBColor(int)
		 */
		int convertDisplayColorToARGBColor(Display display, int displayColor);

		/**
		 * Prepares the blending of two ARGB colors (only useful when the LCD is a palletized LCD).
		 * <p>
		 * By default this feature is not used.
		 *
		 * @param display
		 *            the display widget.
		 * @param foreground
		 *            the foreground ARGB color to convert.
		 * @param background
		 *            the background ARGB color to convert.
		 * @return true when the indexes have been found, false otherwise.
		 * @see LLUIDisplayImpl#prepareBlendingOfIndexedColors(AtomicInteger, AtomicInteger)
		 */
		default boolean prepareBlendingOfIndexedColors(Display display, AtomicInteger foreground,
				AtomicInteger background) {
			return false;
		}
	}

	/**
	 * When the display is double buffered, the backBuffer represents the back buffer where drawings are done. At flush
	 * time, the backBuffer is copied on this display. When the display is simple buffered, the backBuffer and frame
	 * buffer are equals and all drawings are performed in this buffer.
	 */
	protected Image drawingBuffer;

	/**
	 * This buffer contains the application drawings after the call to
	 * {@link #flush(MicroUIGraphicsContext, Image, int, int, int, int)} in double buffer mode or during application
	 * rendering in simple buffer mode.
	 */
	protected Image frameBuffer;

	/**
	 * Buffer visible on front panel viewer: contains application drawings (see {@link #frameBuffer}) and some post
	 * transformations like mask reduction, backlight and contrast.
	 */
	protected Image visibleBuffer;

	// widget extension which describes the display.
	private DisplayExtension extension; // null init
	private String extensionClassName;

	// display size in pixels, may be different than widget size (see setDisplaylWidth())
	private int displayWidth; // 0 init
	private int displayHeight; // 0 init

	// display characteristics
	private int initialColor; // 0 init
	private byte alpha;
	private boolean isDoubleBuffered;
	private boolean hasBacklight;

	// current backlight
	private int backlight;

	/**
	 * Creates the widget display
	 */
	public Display() {

		// fully opaque by default
		this.alpha = (byte) 0xff;

		// double buffer feature enabled by default
		this.isDoubleBuffered = true;

		// backlight feature enabled by default
		this.hasBacklight = true;
	}

	/**
	 * Sets the width of the display in pixels (see class comment).
	 * <p>
	 * This method should only be called by front panel parser.
	 *
	 * @param width
	 *            the width to set.
	 */
	public void setDisplayWidth(int width) {
		if (width < 0) {
			throw new IllegalArgumentException("Display width cannot be negative.");
		}
		this.displayWidth = width;
	}

	/**
	 * Sets the height of the display in pixels (see class comment).
	 * <p>
	 * This method should only be called by front panel parser.
	 *
	 * @param height
	 *            the height to set.
	 */
	public void setDisplayHeight(int height) {
		if (height < 0) {
			throw new IllegalArgumentException("Display height cannot be negative.");
		}
		this.displayHeight = height;
	}

	/**
	 * Sets the initial color of the display (color used on startup, just before first drawing). By default the initial
	 * color is 0x0 (black).
	 * <p>
	 * This is a RGB color (24 bits: 8-8-8). Alpha level is ignored.
	 * <p>
	 * This method should only be called by front panel parser.
	 *
	 * @param initialColor
	 *            the initial color to set.
	 */
	public void setInitialColor(int initialColor) {
		this.initialColor = initialColor | 0xff000000;
	}

	/**
	 * Sets the opacity level (alpha) of the display in order to obtain a translucent display. By default the display is
	 * fully opaque.
	 * <p>
	 * This method should only be called by front panel parser.
	 *
	 * @param alpha
	 *            the opacity level to set, 0x00 is fully transparent and 0xff fully opaque.
	 * @throws IllegalArgumentException
	 *             if the given argument is not between 0x0 and 0xff.
	 */
	public void setAlpha(int alpha) {
		if (alpha < 0x0 || alpha > 0xff) {
			throw new IllegalArgumentException("The opacity level must be a value between 0x0 and 0xff");
		}
		this.alpha = (byte) alpha;
	}

	/**
	 * Enables or disables the double buffering feature.
	 * <p>
	 * By default the double buffering feature is enabled.
	 * <p>
	 * See {@link LLUIDisplayImpl#isDoubleBuffered()} to have more information.
	 *
	 * @param enable
	 *            true to enable double buffering feature (default value).
	 */
	public void setDoubleBufferFeature(boolean enable) {
		this.isDoubleBuffered = enable;
	}

	/**
	 * Enables or disables the backlight feature.
	 * <p>
	 * By default the backlight feature is enabled.
	 * <p>
	 * See {@link LLUIDisplayImpl#hasBacklight()} to have more information.
	 *
	 * @param enable
	 *            true to enable backlight feature (default value).
	 */
	public void setBacklightFeature(boolean enable) {
		this.hasBacklight = enable;
	}

	/**
	 * Defines the optional class name of the display extension. This allows to customize this display widget without
	 * creating a sub-class of {@link Display}.
	 * <p>
	 * When set, a check is performed in {@link #finalizeConfiguration()} to verify whether this class is valid.
	 * <p>
	 * This method should only be called by front panel parser.
	 *
	 * @param extensionClassName
	 *            the user display extension class name.
	 */
	public void setExtensionClass(String extensionClassName) {
		this.extensionClassName = extensionClassName;
	}

	@Override
	public synchronized void finalizeConfiguration() {
		super.finalizeConfiguration();

		// display size may be different than widget size
		if (this.displayWidth == 0) {
			this.displayWidth = getWidth();
		}
		if (this.displayHeight == 0) {
			this.displayHeight = getHeight();
		}

		if (this.extensionClassName != null) {
			// check user class
			FrontPanel.getFrontPanel().verifyUserClass(getClass(), DisplayExtension.class, this.extensionClassName);
		}

		// no need to repaint behind display
		setOverlay(true);
	}

	@Override
	public void showYourself(boolean appearSwitchedOn) {
		Image back = this.drawingBuffer;
		back.fillRectangle(0, 0, back.getWidth(), back.getHeight(), convertDisplayColorToARGBColor(
				convertARGBColorToDisplayColor(appearSwitchedOn ? ~getInitialColor() : getInitialColor())));
		flush(back, 0, 0, back.getWidth(), back.getHeight());
	}

	/**
	 * Called by the parser after filling all the fields defined in the xml. Used to complete the display
	 * initialization.
	 */
	@Override
	public void start() {

		super.start();

		// load optional extension (may load an extension class which can call Device.getDevice(); so this extension
		// cannot be created in finalizeConfiguration() method)
		if (this.extensionClassName != null) {
			// load extension set in fp file
			this.extension = FrontPanel.getFrontPanel().newUserInstance(getClass(), DisplayExtension.class,
					this.extensionClassName);
		}

		// create image returned by getCurrentSkin(); this image may be smaller or higher than display buffers
		this.visibleBuffer = newWidgetImage();

		// create image which will contains the drawings
		this.frameBuffer = newDisplayImage();

		// create image where performing drawings; it is frame buffer in simple buffer mode or a dedicated back buffer
		// in double buffer mode.
		this.drawingBuffer = isDoubleBuffered() ? newDisplayImage() : this.frameBuffer;

		// turn on backlight
		setBacklight(BACKLIGHT_MAX);
	}

	@Override
	public void dispose() {
		FrontPanel.getFrontPanel().disposeIfNotNull(this.frameBuffer, this.visibleBuffer);
		if (this.drawingBuffer != this.frameBuffer) {
			FrontPanel.getFrontPanel().disposeIfNotNull(this.drawingBuffer);
		}
		this.frameBuffer = null;
		this.visibleBuffer = null;
		this.drawingBuffer = null;
		super.dispose();
	}

	/**
	 * Gets the current displayed skin.
	 *
	 * @return the current displayed skin.
	 */
	@Override
	public synchronized Image getCurrentSkin() {

		// draw frame buffer data on visible buffer (scale up/down if necessary)
		Image resultImage = this.visibleBuffer;
		resultImage.drawImage(this.frameBuffer);

		// apply contrast and backlight
		drawContrast(resultImage);
		drawBacklight(resultImage);

		// apply transparency
		byte opacity = this.alpha;
		if (opacity != (byte) 0xff) {
			resultImage = resultImage.getTransparentImage(opacity);
		}

		// crop displayed area according mask
		resultImage.crop(getFilter());

		return resultImage;
	}

	/**
	 * Gets the buffer when drawings are rendered before being copied in frame buffer (call to
	 * {@link #flush(MicroUIGraphicsContext, Image, int, int, int, int)}.
	 *
	 * @return the buffer when performing the drawings
	 */
	public Image getDrawingBuffer() {
		return this.drawingBuffer;
	}

	/**
	 * Gets the initial color of the display.
	 *
	 * @return the initial color.
	 */
	public int getInitialColor() {
		return this.initialColor;
	}

	/**
	 * Called by MicroUI graphical engine when MicroUI.start() is called by the MicroEJ application.
	 */
	@Override
	public Image initialize() {
		// nothing else to initialize, just have to return the back buffer
		return getDrawingBuffer();
	}

	@Override
	public boolean hasBacklight() {
		return this.hasBacklight;
	}

	/**
	 * Returns the backlight of the display. <code>backlight</code> value range is
	 * {@link #BACKLIGHT_MIN}-{@link #BACKLIGHT_MAX}. If the display do not manage backlight (
	 * <code>hasBacklight() == false</code>), returns {@link #BACKLIGHT_MIN}.
	 *
	 * @return the backlight of the display.
	 */
	@Override
	public int getBacklight() {
		return hasBacklight() ? this.backlight : BACKLIGHT_MIN;
	}

	/**
	 * Sets the backlight of the display.
	 *
	 * @param backlight
	 *            the new value of the backlight.
	 */
	@Override
	public void setBacklight(int backlight) {
		if (hasBacklight()) {
			if (this.backlight < BACKLIGHT_MIN) {
				this.backlight = BACKLIGHT_MIN;
			} else if (this.backlight > BACKLIGHT_MAX) {
				this.backlight = BACKLIGHT_MAX;
			} else {
				this.backlight = backlight;
			}
			repaint();
		}
	}

	@Override
	public boolean isDoubleBuffered() {
		return this.isDoubleBuffered;
	}

	@Override
	public boolean isColor() {
		return this.extension != null ? this.extension.isColor(this) : LLUIDisplayImpl.super.isColor();
	}

	@Override
	public int getNumberOfColors() {
		return this.extension != null ? this.extension.getNumberOfColors(this)
				: LLUIDisplayImpl.super.getNumberOfColors();
	}

	@Override
	public int convertARGBColorToDisplayColor(int argbColor) {
		return this.extension != null ? this.extension.convertARGBColorToDisplayColor(this, argbColor)
				: LLUIDisplayImpl.super.convertARGBColorToDisplayColor(argbColor);
	}

	@Override
	public int convertDisplayColorToARGBColor(int displayColor) {
		return this.extension != null ? this.extension.convertDisplayColorToARGBColor(this, displayColor)
				: LLUIDisplayImpl.super.convertDisplayColorToARGBColor(displayColor);
	}

	@Override
	public boolean prepareBlendingOfIndexedColors(AtomicInteger foreground, AtomicInteger background) {
		return this.extension != null ? this.extension.prepareBlendingOfIndexedColors(this, foreground, background)
				: LLUIDisplayImpl.super.prepareBlendingOfIndexedColors(foreground, background);
	}

	@Override
	public void flush(MicroUIGraphicsContext gc, Image image, int x, int y, int width, int height) {
		flush(image, x, y, width, height);
	}

	private void flush(Image image, int x, int y, int width, int height) {
		if (isDoubleBuffered()) {
			// draw back buffer data in frame buffer
			this.frameBuffer.drawImage(image, x, y, width, height, x, y, width, height);
		}

		// launch repaint
		if (this.alpha != (byte) 0xff) {
			// parent must repaint the display background
			getParent().repaint(getX(), getY(), getWidth(), getHeight());
		} else {
			// all display should be repaint because the realSize and the size
			// can be different
			repaint();
		}
	}

	/**
	 * Creates a new image with the same size than display (not widget size).
	 *
	 * @return an image as big as display.
	 */
	protected Image newDisplayImage() {
		return FrontPanel.getFrontPanel().newImage(this.displayWidth, this.displayHeight, this.initialColor, true);
	}

	/**
	 * Creates a new image with the same size than widget (not display size).
	 *
	 * @return an image as big as widget.
	 */
	protected Image newWidgetImage() {
		return FrontPanel.getFrontPanel().newImage(getWidth(), getHeight(), this.initialColor, true);
	}

	/**
	 * Transforms the skin data applying an algorithm to simulate the display contrast.
	 * <p>
	 * The default implementation performs nothing.
	 *
	 * @param imageSkin
	 *            the image where applying the algorithm.
	 */
	protected void drawContrast(Image imageSkin) {
		// not implemented
	}

	/**
	 * Transforms the skin data applying an algorithm to simulate the display backlight.
	 *
	 * @param imageSkin
	 *            the image where applying the algorithm.
	 */
	protected void drawBacklight(Image imageSkin) {
		if (hasBacklight()) {
			// get a transparency level according backlight
			int shadow = ((BACKLIGHT_MAX - getBacklight()) * 0xff / BACKLIGHT_MAX);
			imageSkin.fillTransparentRectangle(0, 0, imageSkin.getWidth(), imageSkin.getHeight(), shadow << 24);
		}
	}
}
