/*
 * Java
 *
 * Copyright 2019 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.bluetooth.util.server;

import ej.annotation.NonNull;
import ej.basictool.ArrayTools;
import ej.bluetooth.BluetoothCharacteristic;
import ej.bluetooth.BluetoothDevice;
import ej.bluetooth.BluetoothProperties;
import ej.bluetooth.BluetoothUuid;

/**
 * A {@link BluetoothCharacteristic} that can be notified and receive read and write request.
 */
public class BasicServerCharacteristic extends BluetoothCharacteristic implements NotificationCallback {

	private BasicServerDescriptor[] descriptors;
	private CCCDescriptor cccDescriptor;

	/**
	 * Instantiates a {@link BasicServerCharacteristic}. If the characteristic can be notified or indicate, adds a CCC
	 * descriptor.
	 *
	 * @param uuid
	 *            the uuid of the characteristic.
	 * @param properties
	 *            the properties of the characteristic.
	 * @param permissions
	 *            the permission.
	 */
	public BasicServerCharacteristic(@NonNull BluetoothUuid uuid, int properties, int permissions) {
		super(uuid, properties, permissions);
		this.descriptors = new BasicServerDescriptor[0];

		if ((properties & (BluetoothProperties.NOTIFY | BluetoothProperties.INDICATE)) != 0) {
			this.cccDescriptor = new CCCDescriptor();
			addDescriptor(this.cccDescriptor);
		}
	}

	/**
	 * Add a descriptor.
	 *
	 * @param descriptor
	 *            the descriptor to add.
	 */
	public void addDescriptor(@NonNull BasicServerDescriptor descriptor) {
		super.addDescriptor(descriptor);
		this.descriptors = ArrayTools.add(this.descriptors, descriptor);
	}

	/**
	 * Gets the server descriptor.
	 *
	 * @return the descriptor.
	 */
	public BasicServerDescriptor[] getServerDescriptors() {
		return this.descriptors;
	}

	@Override
	public void onWriteRequest(BluetoothDevice device, byte[] value) {
		// Nothing to do.

	}

	@Override
	public void onReadRequest(BluetoothDevice device) {
		// Nothing to do.
	}

	@Override
	public void onNotify(BluetoothDevice device) {
		// Nothing to do.
	}

	/**
	 * Checks whether a device should be notified.
	 *
	 * @param device
	 *            the device.
	 * @return <code>true</code> if the device should be notified.
	 */
	private boolean shouldNotify(BluetoothDevice device) {
		return this.cccDescriptor != null && this.cccDescriptor.shouldNotify(device);
	}

	/**
	 * Notify devices.
	 *
	 * @param devices
	 *            the devices to notify.
	 */
	/** package **/
	void notify(BluetoothDevice[] devices) {
		for (BluetoothDevice bluetoothDevice : devices) {
			if (shouldNotify(bluetoothDevice)) {
				onNotify(bluetoothDevice);
			}
		}
	}
}
