/*
 * Java
 *
 * Copyright 2017 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package java.math;

/**
 * Provides missing methods in java.lang.Long.
 */
class LongHelper {

	/**
	 * Returns the number of zero bits preceding the highest-order ("leftmost") one-bit in the two's complement binary
	 * representation of the specified {@code long} value. Returns 64 if the specified value has no one-bits in its
	 * two's complement representation, in other words if it is equal to zero.
	 *
	 * <p>
	 * Note that this method is closely related to the logarithm base 2. For all positive {@code long} values x:
	 * <ul>
	 * <li>floor(log<sub>2</sub>(x)) = {@code 63 - numberOfLeadingZeros(x)}
	 * <li>ceil(log<sub>2</sub>(x)) = {@code 64 - numberOfLeadingZeros(x - 1)}
	 * </ul>
	 *
	 * @param i
	 *            the value whose number of leading zeros is to be computed
	 * @return the number of zero bits preceding the highest-order ("leftmost") one-bit in the two's complement binary
	 *         representation of the specified {@code long} value, or 64 if the value is equal to zero.
	 * @since 1.5
	 */
	static int numberOfLeadingZeros(long i) {
		// HD, Figure 5-6
		if (i == 0) {
			return 64;
		}
		int n = 1;
		int x = (int) (i >>> 32);
		if (x == 0) {
			n += 32;
			x = (int) i;
		}
		if (x >>> 16 == 0) {
			n += 16;
			x <<= 16;
		}
		if (x >>> 24 == 0) {
			n += 8;
			x <<= 8;
		}
		if (x >>> 28 == 0) {
			n += 4;
			x <<= 4;
		}
		if (x >>> 30 == 0) {
			n += 2;
			x <<= 2;
		}
		n -= x >>> 31;
		return n;
	}
}
