/*
 * Java
 *
 * Copyright 2012-2025 MicroEJ Corp.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.microui.event.generator;

import ej.annotation.Nullable;
import ej.microui.display.Display;
import ej.microui.event.Event;
import ej.microui.event.EventGenerator;
import ej.microui.event.EventHandler;

/**
 * Generic event generator.
 * <p>
 * Generic communication to receive events.
 *
 * @since 2.0
 */
public abstract class GenericEventGenerator extends EventGenerator {

	/**
	 * Called at startup to configure the event generator with the specific properties set in the MicroUI XML file.
	 *
	 * @param name
	 *            the property name
	 * @param value
	 *            the property value
	 */
	public abstract void setProperty(String name, String value);

	/**
	 * Called by MicroUI pump when a custom event has been received from the native world. The event contains only one
	 * 32-bit value.
	 *
	 * Implementation has to decode the event. Then it can
	 * <ul>
	 * <li>perform any actions according the event (for instance update a model, etc.),</li>
	 * <li>or save it, build a standard MicroUI event (see {@link Event#buildEvent(int, EventGenerator, int)} and send
	 * this event to the given event handler.</li>
	 * </ul>
	 * <p>
	 * The second solution is useful to deport the event management to a global events handler like other events
	 * (command, buttons, pointer and states).
	 * <p>
	 * When the handler is the display's event serializer (see {@link Display#getEventHandler()}) the given handler is
	 * the current displayable; it allows to handle the new MicroUI event immediately, without postponing it to the end
	 * of the events queue. Otherwise the given handler is the handler returned by {@link #getEventHandler()}.
	 * <p>
	 * The handler may be <code>null</code> when there is no current displayable or when no handler is set on the event
	 * generator.
	 * <p>
	 * The default implementation calls {@link #eventReceived(int)}.
	 *
	 * @param event
	 *            the 32-bit custom event value.
	 * @param handler
	 *            the generator's events handler: the current displayable on display, the handler returned by
	 *            {@link #getEventHandler()} or <code>null</code>.
	 */
	protected void eventReceived(int event, @Nullable EventHandler handler) {
		throw new RuntimeException();
	}

	/**
	 * Called by the default implementation of {@link #eventReceived(int, EventHandler)} when a custom event has been
	 * received from the native world. The event contains only one 32-bit value.
	 *
	 * @param event
	 *            the 32-bit custom event value.
	 */
	protected abstract void eventReceived(int event);

	/**
	 * Called by MicroUI pump when a custom event has been received from the native world. The event contains several
	 * 32-bit values.
	 * <p>
	 * See {@link #eventReceived(int, EventHandler)} comment.
	 * <p>
	 * The default implementation calls {@link #eventsReceived(int[])}.
	 *
	 * @param events
	 *            the 32-bit custom event values.
	 * @param handler
	 *            the generator's events handler: the current displayable on display, the handler returned by
	 *            {@link #getEventHandler()} or <code>null</code>.
	 */
	protected void eventsReceived(int[] events, @Nullable EventHandler handler) {
		throw new RuntimeException();
	}

	/**
	 * Called by the default implementation of {@link #eventReceived(int, EventHandler)} when a custom event has been
	 * received from the native world. The event contains several 32-bit values.
	 *
	 * @param events
	 *            the 32-bit custom event values.
	 * @see #eventReceived(int)
	 */
	protected abstract void eventsReceived(int[] events);
}
