/*
 * Copyright (c) 1995, 2013, Oracle and/or its affiliates. All rights reserved.
 * Copyright 2015-2022 MicroEJ Corp. This file has been modified and/or created by MicroEJ Corp.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation. The Copyright holder(s) designates
 * this particular file as subject to the "Classpath" exception as provided
 * by the Copyright holder(s) in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact the Copyright holder(s) if you need additional information
 * or have any questions.
 * (if Oracle: 500 Oracle Parkway, Redwood Shores, CA 94065 USA or visit www.oracle.com)
 */

package java.net;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import ej.annotation.Nullable;

/**
 * This class implements client sockets (also called just "sockets"). A socket is an endpoint for communication between
 * two machines.
 *
 * @author unascribed
 * @since JDK1.0
 */
public class Socket implements java.io.Closeable {
	/**
	 * Creates an unconnected socket, with the system-default type of SocketImpl.
	 *
	 * @since JDK1.1
	 * @revised 1.4
	 */
	public Socket() {
		throw new RuntimeException();
	}

	/**
	 * Creates a stream socket and connects it to the specified port number on the named host.
	 * <p>
	 * If the specified host is {@code null} it is the equivalent of specifying the address as
	 * {@link java.net.InetAddress#getByName InetAddress.getByName}{@code (null)}. In other words, it is equivalent to
	 * specifying an address of the loopback interface.
	 * </p>
	 * <p>
	 * If the application has specified a server socket factory, that factory's {@code createSocketImpl} method is
	 * called to create the actual socket implementation. Otherwise a "plain" socket is created.
	 * <p>
	 * If there is a security manager, its {@code checkConnect} method is called with the host address and {@code port}
	 * as its arguments. This could result in a SecurityException.
	 *
	 * @param host
	 *            the host name, or {@code null} for the loopback address.
	 * @param port
	 *            the port number.
	 *
	 * @exception UnknownHostException
	 *                if the IP address of the host could not be determined.
	 *
	 * @exception IOException
	 *                if an I/O error occurs when creating the socket.
	 * @exception SecurityException
	 *                if a security manager exists and its {@code checkConnect} method doesn't allow the operation.
	 * @exception IllegalArgumentException
	 *                if the port parameter is outside the specified range of valid port values, which is between 0 and
	 *                65535, inclusive.
	 */
	public Socket(@Nullable String host, int port) throws UnknownHostException, IOException {
		throw new RuntimeException();
	}

	/**
	 * Creates a stream socket and connects it to the specified port number at the specified IP address.
	 * <p>
	 * If the application has specified a socket factory, that factory's {@code createSocketImpl} method is called to
	 * create the actual socket implementation. Otherwise a "plain" socket is created.
	 * <p>
	 * If there is a security manager, its {@code checkConnect} method is called with the host address and {@code port}
	 * as its arguments. This could result in a SecurityException.
	 *
	 * @param address
	 *            the IP address.
	 * @param port
	 *            the port number.
	 * @exception IOException
	 *                if an I/O error occurs when creating the socket.
	 * @exception SecurityException
	 *                if a security manager exists and its {@code checkConnect} method doesn't allow the operation.
	 * @exception IllegalArgumentException
	 *                if the port parameter is outside the specified range of valid port values, which is between 0 and
	 *                65535, inclusive.
	 * @exception NullPointerException
	 *                if {@code address} is null.
	 */
	public Socket(InetAddress address, int port) throws IOException {
		throw new RuntimeException();
	}

	/**
	 * Creates a socket and connects it to the specified remote host on the specified remote port. The Socket will also
	 * bind() to the local address and port supplied.
	 * <p>
	 * If the specified host is {@code null} it is the equivalent of specifying the address as
	 * {@link java.net.InetAddress#getByName InetAddress.getByName}{@code (null)}. In other words, it is equivalent to
	 * specifying an address of the loopback interface.
	 * </p>
	 * <p>
	 * A local port number of {@code zero} will let the system pick up a free port in the {@code bind} operation.
	 * </p>
	 * <p>
	 * If there is a security manager, its {@code checkConnect} method is called with the host address and {@code port}
	 * as its arguments. This could result in a SecurityException.
	 *
	 * @param host
	 *            the name of the remote host, or {@code null} for the loopback address.
	 * @param port
	 *            the remote port
	 * @param localAddr
	 *            the local address the socket is bound to, or {@code null} for the {@code anyLocal} address.
	 * @param localPort
	 *            the local port the socket is bound to, or {@code zero} for a system selected free port.
	 * @exception IOException
	 *                if an I/O error occurs when creating the socket.
	 * @exception SecurityException
	 *                if a security manager exists and its {@code checkConnect} method doesn't allow the connection to
	 *                the destination, or if its {@code checkListen} method doesn't allow the bind to the local port.
	 * @exception IllegalArgumentException
	 *                if the port parameter or localPort parameter is outside the specified range of valid port values,
	 *                which is between 0 and 65535, inclusive.
	 * @since JDK1.1
	 */
	public Socket(@Nullable String host, int port, @Nullable InetAddress localAddr, int localPort) throws IOException {
		throw new RuntimeException();
	}

	/**
	 * Creates a socket and connects it to the specified remote address on the specified remote port. The Socket will
	 * also bind() to the local address and port supplied.
	 * <p>
	 * If the specified local address is {@code null} it is the equivalent of specifying the address as the AnyLocal
	 * address (see {@link java.net.InetAddress#isAnyLocalAddress InetAddress.isAnyLocalAddress}{@code ()}).
	 * <p>
	 * A local port number of {@code zero} will let the system pick up a free port in the {@code bind} operation.
	 * </p>
	 * <p>
	 * If there is a security manager, its {@code checkConnect} method is called with the host address and {@code port}
	 * as its arguments. This could result in a SecurityException.
	 *
	 * @param address
	 *            the remote address
	 * @param port
	 *            the remote port
	 * @param localAddr
	 *            the local address the socket is bound to, or {@code null} for the {@code anyLocal} address.
	 * @param localPort
	 *            the local port the socket is bound to or {@code zero} for a system selected free port.
	 * @exception IOException
	 *                if an I/O error occurs when creating the socket.
	 * @exception SecurityException
	 *                if a security manager exists and its {@code checkConnect} method doesn't allow the connection to
	 *                the destination, or if its {@code checkListen} method doesn't allow the bind to the local port.
	 * @exception IllegalArgumentException
	 *                if the port parameter or localPort parameter is outside the specified range of valid port values,
	 *                which is between 0 and 65535, inclusive.
	 * @exception NullPointerException
	 *                if {@code address} is null.
	 * @since JDK1.1
	 */
	public Socket(InetAddress address, int port, @Nullable InetAddress localAddr, int localPort) throws IOException {
		throw new RuntimeException();
	}

	/**
	 * Connects this socket to the server.
	 *
	 * @param endpoint
	 *            the {@code SocketAddress}
	 * @throws IOException
	 *             if an error occurs during the connection
	 * @throws IllegalArgumentException
	 *             if endpoint is null or is a SocketAddress subclass not supported by this socket
	 * @since 1.4
	 * @spec JSR-51
	 */
	public void connect(SocketAddress endpoint) throws IOException {
		throw new RuntimeException();
	}

	/**
	 * Connects this socket to the server with a specified timeout value. A timeout of zero is interpreted as an
	 * infinite timeout. The connection will then block until established or an error occurs.
	 *
	 * @param endpoint
	 *            the {@code SocketAddress}
	 * @param timeout
	 *            the timeout value to be used in milliseconds.
	 * @throws IOException
	 *             if an error occurs during the connection
	 * @throws SocketTimeoutException
	 *             if timeout expires before connecting
	 * @throws IllegalArgumentException
	 *             if endpoint is null or is a SocketAddress subclass not supported by this socket
	 * @since 1.4
	 * @spec JSR-51
	 */
	public void connect(SocketAddress endpoint, int timeout) throws IOException {
		throw new RuntimeException();
	}

	/**
	 * Binds the socket to a local address.
	 * <P>
	 * If the address is {@code null}, then the system will pick up an ephemeral port and a valid local address to bind
	 * the socket.
	 *
	 * @param bindpoint
	 *            the {@code SocketAddress} to bind to
	 * @throws IOException
	 *             if the bind operation fails, or if the socket is already bound.
	 * @throws IllegalArgumentException
	 *             if bindpoint is a SocketAddress subclass not supported by this socket
	 * @throws SecurityException
	 *             if a security manager exists and its {@code checkListen} method doesn't allow the bind to the local
	 *             port.
	 *
	 * @since 1.4
	 * @see #isBound
	 */
	public void bind(@Nullable SocketAddress bindpoint) throws IOException {
		throw new RuntimeException();
	}

	/**
	 * Returns the address to which the socket is connected.
	 * <p>
	 * If the socket was connected prior to being {@link #close closed}, then this method will continue to return the
	 * connected address after the socket is closed.
	 *
	 * @return the remote IP address to which this socket is connected, or {@code null} if the socket is not connected.
	 */
	@Nullable
	public InetAddress getInetAddress() {
		throw new RuntimeException();
	}

	/**
	 * Gets the local address to which the socket is bound.
	 * <p>
	 * If there is a security manager set, its {@code checkConnect} method is called with the local address and
	 * {@code -1} as its arguments to see if the operation is allowed.
	 *
	 * @return the local address to which the socket is bound, or the wildcard address if the socket is closed or not
	 *         bound yet.
	 * @since JDK1.1
	 *
	 */
	public InetAddress getLocalAddress() {
		throw new RuntimeException();
	}

	/**
	 * Returns the remote port number to which this socket is connected.
	 * <p>
	 * If the socket was connected prior to being {@link #close closed}, then this method will continue to return the
	 * connected port number after the socket is closed.
	 *
	 * @return the remote port number to which this socket is connected, or 0 if the socket is not connected yet.
	 */
	public int getPort() {
		throw new RuntimeException();
	}

	/**
	 * Returns the local port number to which this socket is bound.
	 * <p>
	 * If the socket was bound prior to being {@link #close closed}, then this method will continue to return the local
	 * port number after the socket is closed.
	 *
	 * @return the local port number to which this socket is bound or -1 if the socket is not bound yet.
	 */
	public int getLocalPort() {
		throw new RuntimeException();
	}

	/**
	 * Returns the address of the endpoint this socket is connected to, or {@code null} if it is unconnected.
	 * <p>
	 * If the socket was connected prior to being {@link #close closed}, then this method will continue to return the
	 * connected address after the socket is closed.
	 *
	 *
	 * @return a {@code SocketAddress} representing the remote endpoint of this socket, or {@code null} if it is not
	 *         connected yet.
	 * @see #getInetAddress()
	 * @see #getPort()
	 * @see #connect(SocketAddress, int)
	 * @see #connect(SocketAddress)
	 * @since 1.4
	 */
	@Nullable
	public SocketAddress getRemoteSocketAddress() {
		throw new RuntimeException();
	}

	/**
	 * Returns the address of the endpoint this socket is bound to.
	 * <p>
	 * If a socket bound to an endpoint represented by an {@code InetSocketAddress } is {@link #close closed}, then this
	 * method will continue to return an {@code InetSocketAddress} after the socket is closed. In that case the returned
	 * {@code InetSocketAddress}'s address is the {@link InetAddress#isAnyLocalAddress wildcard} address and its port is
	 * the local port that it was bound to.
	 * <p>
	 * If there is a security manager set, its {@code checkConnect} method is called with the local address and
	 * {@code -1} as its arguments to see if the operation is allowed.
	 *
	 * @return a {@code SocketAddress} representing the local endpoint of this socket, or {@code null} if the socket is
	 *         not bound yet.
	 *
	 * @see #getLocalAddress()
	 * @see #getLocalPort()
	 * @see #bind(SocketAddress)
	 * @since 1.4
	 */
	@Nullable
	public SocketAddress getLocalSocketAddress() {
		throw new RuntimeException();
	}

	/**
	 * Returns an input stream for this socket.
	 *
	 * <p>
	 * Under abnormal conditions the underlying connection may be broken by the remote host or the network software (for
	 * example a connection reset in the case of TCP connections). When a broken connection is detected by the network
	 * software the following applies to the returned input stream :-
	 *
	 * <ul>
	 *
	 * <li>
	 * <p>
	 * The network software may discard bytes that are buffered by the socket. Bytes that aren't discarded by the
	 * network software can be read using {@link java.io.InputStream#read read}.
	 *
	 * <li>
	 * <p>
	 * If there are no bytes buffered on the socket, or all buffered bytes have been consumed by
	 * {@link java.io.InputStream#read read}, then all subsequent calls to {@link java.io.InputStream#read read} will
	 * throw an {@link java.io.IOException IOException}.
	 *
	 * <li>
	 * <p>
	 * If there are no bytes buffered on the socket, and the socket has not been closed using {@link #close close}, then
	 * {@link java.io.InputStream#available available} will return {@code 0}.
	 *
	 * </ul>
	 *
	 * <p>
	 * Closing the returned {@link java.io.InputStream InputStream} will close the associated socket.
	 *
	 * @return an input stream for reading bytes from this socket.
	 * @exception IOException
	 *                if an I/O error occurs when creating the input stream, the socket is closed or the socket is not
	 *                connected.
	 *
	 * @revised 1.4
	 * @spec JSR-51
	 */
	public InputStream getInputStream() throws IOException {
		throw new RuntimeException();
	}

	/**
	 * Returns an output stream for this socket.
	 *
	 *
	 * <p>
	 * Closing the returned {@link java.io.OutputStream OutputStream} will close the associated socket.
	 *
	 * @return an output stream for writing bytes to this socket.
	 * @exception IOException
	 *                if an I/O error occurs when creating the output stream or if the socket is not connected.
	 * @revised 1.4
	 * @spec JSR-51
	 */
	public OutputStream getOutputStream() throws IOException {
		throw new RuntimeException();
	}

	/**
	 * Enable/disable {@link SocketOptions#TCP_NODELAY TCP_NODELAY} (disable/enable Nagle's algorithm).
	 *
	 * @param on
	 *            {@code true} to enable TCP_NODELAY, {@code false} to disable.
	 *
	 * @exception SocketException
	 *                if there is an error in the underlying protocol, such as a TCP error.
	 *
	 * @since JDK1.1
	 *
	 * @see #getTcpNoDelay()
	 */
	public void setTcpNoDelay(boolean on) throws SocketException {
		throw new RuntimeException();
	}

	/**
	 * Tests if {@link SocketOptions#TCP_NODELAY TCP_NODELAY} is enabled.
	 *
	 * @return a {@code boolean} indicating whether or not {@link SocketOptions#TCP_NODELAY TCP_NODELAY} is enabled.
	 * @exception SocketException
	 *                if there is an error in the underlying protocol, such as a TCP error.
	 * @since JDK1.1
	 * @see #setTcpNoDelay(boolean)
	 */
	public boolean getTcpNoDelay() throws SocketException {
		throw new RuntimeException();
	}

	/**
	 * Enable/disable {@link SocketOptions#SO_LINGER SO_LINGER} with the specified linger time in seconds. The maximum
	 * timeout value is platform specific.
	 *
	 * The setting only affects socket close.
	 *
	 * @param on
	 *            whether or not to linger on.
	 * @param linger
	 *            how long to linger for, if on is true.
	 * @exception SocketException
	 *                if there is an error in the underlying protocol, such as a TCP error.
	 * @exception IllegalArgumentException
	 *                if the linger value is negative.
	 * @since JDK1.1
	 * @see #getSoLinger()
	 */
	public void setSoLinger(boolean on, int linger) throws SocketException {
		throw new RuntimeException();
	}

	/**
	 * Returns setting for {@link SocketOptions#SO_LINGER SO_LINGER}. -1 returns implies that the option is disabled.
	 *
	 * The setting only affects socket close.
	 *
	 * @return the setting for {@link SocketOptions#SO_LINGER SO_LINGER}.
	 * @exception SocketException
	 *                if there is an error in the underlying protocol, such as a TCP error.
	 * @since JDK1.1
	 * @see #setSoLinger(boolean, int)
	 */
	public int getSoLinger() throws SocketException {
		throw new RuntimeException();
	}

	/**
	 * Enable/disable {@link SocketOptions#SO_OOBINLINE SO_OOBINLINE} (receipt of TCP urgent data)
	 *
	 * By default, this option is disabled and TCP urgent data received on a socket is silently discarded. If the user
	 * wishes to receive urgent data, then this option must be enabled. When enabled, urgent data is received inline
	 * with normal data.
	 * <p>
	 * Note, only limited support is provided for handling incoming urgent data. In particular, no notification of
	 * incoming urgent data is provided and there is no capability to distinguish between normal data and urgent data
	 * unless provided by a higher level protocol.
	 *
	 * @param on
	 *            {@code true} to enable {@link SocketOptions#SO_OOBINLINE SO_OOBINLINE}, {@code false} to disable.
	 *
	 * @exception SocketException
	 *                if there is an error in the underlying protocol, such as a TCP error.
	 *
	 * @since 1.4
	 *
	 * @see #getOOBInline()
	 */
	public void setOOBInline(boolean on) throws SocketException {
		throw new RuntimeException();
	}

	/**
	 * Tests if {@link SocketOptions#SO_OOBINLINE SO_OOBINLINE} is enabled.
	 *
	 * @return a {@code boolean} indicating whether or not {@link SocketOptions#SO_OOBINLINE SO_OOBINLINE}is enabled.
	 *
	 * @exception SocketException
	 *                if there is an error in the underlying protocol, such as a TCP error.
	 * @since 1.4
	 * @see #setOOBInline(boolean)
	 */
	public boolean getOOBInline() throws SocketException {
		throw new RuntimeException();
	}

	/**
	 * Enable/disable {@link SocketOptions#SO_TIMEOUT SO_TIMEOUT} with the specified timeout, in milliseconds. With this
	 * option set to a non-zero timeout, a read() call on the InputStream associated with this Socket will block for
	 * only this amount of time. If the timeout expires, a <B>java.net.SocketTimeoutException</B> is raised, though the
	 * Socket is still valid. The option <B>must</B> be enabled prior to entering the blocking operation to have effect.
	 * The timeout must be {@code > 0}. A timeout of zero is interpreted as an infinite timeout.
	 *
	 * @param timeout
	 *            the specified timeout, in milliseconds.
	 * @exception SocketException
	 *                if there is an error in the underlying protocol, such as a TCP error.
	 * @since JDK 1.1
	 * @see #getSoTimeout()
	 */
	public synchronized void setSoTimeout(int timeout) throws SocketException {
		throw new RuntimeException();
	}

	/**
	 * Returns setting for {@link SocketOptions#SO_TIMEOUT SO_TIMEOUT}. 0 returns implies that the option is disabled
	 * (i.e., timeout of infinity).
	 *
	 * @return the setting for {@link SocketOptions#SO_TIMEOUT SO_TIMEOUT}
	 * @exception SocketException
	 *                if there is an error in the underlying protocol, such as a TCP error.
	 *
	 * @since JDK1.1
	 * @see #setSoTimeout(int)
	 */
	public synchronized int getSoTimeout() throws SocketException {
		throw new RuntimeException();
	}

	/**
	 * Sets the {@link SocketOptions#SO_SNDBUF SO_SNDBUF} option to the specified value for this {@code Socket}. The
	 * {@link SocketOptions#SO_SNDBUF SO_SNDBUF} option is used by the platform's networking code as a hint for the size
	 * to set the underlying network I/O buffers.
	 *
	 * <p>
	 * Because {@link SocketOptions#SO_SNDBUF SO_SNDBUF} is a hint, applications that want to verify what size the
	 * buffers were set to should call {@link #getSendBufferSize()}.
	 *
	 * @exception SocketException
	 *                if there is an error in the underlying protocol, such as a TCP error.
	 *
	 * @param size
	 *            the size to which to set the send buffer size. This value must be greater than 0.
	 *
	 * @exception IllegalArgumentException
	 *                if the value is 0 or is negative.
	 *
	 * @see #getSendBufferSize()
	 * @since 1.2
	 */
	public synchronized void setSendBufferSize(int size) throws SocketException {
		throw new RuntimeException();
	}

	/**
	 * Get value of the {@link SocketOptions#SO_SNDBUF SO_SNDBUF} option for this {@code Socket}, that is the buffer
	 * size used by the platform for output on this {@code Socket}.
	 *
	 * @return the value of the {@link SocketOptions#SO_SNDBUF SO_SNDBUF} option for this {@code Socket}.
	 *
	 * @exception SocketException
	 *                if there is an error in the underlying protocol, such as a TCP error.
	 *
	 * @see #setSendBufferSize(int)
	 * @since 1.2
	 */
	public synchronized int getSendBufferSize() throws SocketException {
		throw new RuntimeException();
	}

	/**
	 * Sets the {@link SocketOptions#SO_RCVBUF SO_RCVBUF} option to the specified value for this {@code Socket}. The
	 * {@link SocketOptions#SO_RCVBUF SO_RCVBUF} option is used by the platform's networking code as a hint for the size
	 * to set the underlying network I/O buffers.
	 *
	 * <p>
	 * Increasing the receive buffer size can increase the performance of network I/O for high-volume connection, while
	 * decreasing it can help reduce the backlog of incoming data.
	 *
	 * <p>
	 * Because {@link SocketOptions#SO_RCVBUF SO_RCVBUF} is a hint, applications that want to verify what size the
	 * buffers were set to should call {@link #getReceiveBufferSize()}.
	 *
	 * <p>
	 * The value of {@link SocketOptions#SO_RCVBUF SO_RCVBUF} is also used to set the TCP receive window that is
	 * advertized to the remote peer. Generally, the window size can be modified at any time when a socket is connected.
	 * However, if a receive window larger than 64K is required then this must be requested <B>before</B> the socket is
	 * connected to the remote peer. There are two cases to be aware of:
	 * <ol>
	 * <li>For sockets accepted from a ServerSocket, this must be done by calling
	 * {@link ServerSocket#setReceiveBufferSize(int)} before the ServerSocket is bound to a local address.
	 * <p>
	 * </li>
	 * <li>For client sockets, setReceiveBufferSize() must be called before connecting the socket to its remote
	 * peer.</li>
	 * </ol>
	 *
	 * @param size
	 *            the size to which to set the receive buffer size. This value must be greater than 0.
	 *
	 * @exception IllegalArgumentException
	 *                if the value is 0 or is negative.
	 *
	 * @exception SocketException
	 *                if there is an error in the underlying protocol, such as a TCP error.
	 *
	 * @see #getReceiveBufferSize()
	 * @see ServerSocket#setReceiveBufferSize(int)
	 * @since 1.2
	 */
	public synchronized void setReceiveBufferSize(int size) throws SocketException {
		throw new RuntimeException();
	}

	/**
	 * Gets the value of the {@link SocketOptions#SO_RCVBUF SO_RCVBUF} option for this {@code Socket}, that is the
	 * buffer size used by the platform for input on this {@code Socket}.
	 *
	 * @return the value of the {@link SocketOptions#SO_RCVBUF SO_RCVBUF} option for this {@code Socket}.
	 * @exception SocketException
	 *                if there is an error in the underlying protocol, such as a TCP error.
	 * @see #setReceiveBufferSize(int)
	 * @since 1.2
	 */
	public synchronized int getReceiveBufferSize() throws SocketException {
		throw new RuntimeException();
	}

	/**
	 * Enable/disable {@link SocketOptions#SO_KEEPALIVE SO_KEEPALIVE}.
	 *
	 * @param on
	 *            whether or not to have socket keep alive turned on.
	 * @exception SocketException
	 *                if there is an error in the underlying protocol, such as a TCP error.
	 * @since 1.3
	 * @see #getKeepAlive()
	 */
	public void setKeepAlive(boolean on) throws SocketException {
		throw new RuntimeException();
	}

	/**
	 * Tests if {@link SocketOptions#SO_KEEPALIVE SO_KEEPALIVE} is enabled.
	 *
	 * @return a {@code boolean} indicating whether or not {@link SocketOptions#SO_KEEPALIVE SO_KEEPALIVE} is enabled.
	 * @exception SocketException
	 *                if there is an error in the underlying protocol, such as a TCP error.
	 * @since 1.3
	 * @see #setKeepAlive(boolean)
	 */
	public boolean getKeepAlive() throws SocketException {
		throw new RuntimeException();
	}

	/**
	 * Sets traffic class or type-of-service octet in the IP header for packets sent from this Socket. As the underlying
	 * network implementation may ignore this value applications should consider it a hint.
	 *
	 * <P>
	 * The tc <B>must</B> be in the range {@code 0 <= tc <=
	 * 255} or an IllegalArgumentException will be thrown.
	 * <p>
	 * Notes:
	 * <p>
	 * For Internet Protocol v4 the value consists of an {@code integer}, the least significant 8 bits of which
	 * represent the value of the TOS octet in IP packets sent by the socket. RFC 1349 defines the TOS values as
	 * follows:
	 *
	 * <UL>
	 * <LI><CODE>IPTOS_LOWCOST (0x02)</CODE></LI>
	 * <LI><CODE>IPTOS_RELIABILITY (0x04)</CODE></LI>
	 * <LI><CODE>IPTOS_THROUGHPUT (0x08)</CODE></LI>
	 * <LI><CODE>IPTOS_LOWDELAY (0x10)</CODE></LI>
	 * </UL>
	 * The last low order bit is always ignored as this corresponds to the MBZ (must be zero) bit.
	 * <p>
	 * Setting bits in the precedence field may result in a SocketException indicating that the operation is not
	 * permitted.
	 * <p>
	 * As RFC 1122 section 4.2.4.2 indicates, a compliant TCP implementation should, but is not required to, let
	 * application change the TOS field during the lifetime of a connection. So whether the type-of-service field can be
	 * changed after the TCP connection has been established depends on the implementation in the underlying platform.
	 * Applications should not assume that they can change the TOS field after the connection.
	 * <p>
	 * For Internet Protocol v6 {@code tc} is the value that would be placed into the sin6_flowinfo field of the IP
	 * header.
	 *
	 * @param tc
	 *            an {@code int} value for the bitset.
	 * @throws SocketException
	 *             if there is an error setting the traffic class or type-of-service
	 * @since 1.4
	 * @see #getTrafficClass
	 * @see SocketOptions#IP_TOS
	 */
	public void setTrafficClass(int tc) throws SocketException {
		throw new RuntimeException();
	}

	/**
	 * Gets traffic class or type-of-service in the IP header for packets sent from this Socket
	 * <p>
	 * As the underlying network implementation may ignore the traffic class or type-of-service set using
	 * {@link #setTrafficClass(int)} this method may return a different value than was previously set using the
	 * {@link #setTrafficClass(int)} method on this Socket.
	 *
	 * @return the traffic class or type-of-service already set
	 * @throws SocketException
	 *             if there is an error obtaining the traffic class or type-of-service value.
	 * @since 1.4
	 * @see #setTrafficClass(int)
	 * @see SocketOptions#IP_TOS
	 */
	public int getTrafficClass() throws SocketException {
		throw new RuntimeException();
	}

	/**
	 * Enable/disable the {@link SocketOptions#SO_REUSEADDR SO_REUSEADDR} socket option.
	 * <p>
	 * When a TCP connection is closed the connection may remain in a timeout state for a period of time after the
	 * connection is closed (typically known as the {@code TIME_WAIT} state or {@code 2MSL} wait state). For
	 * applications using a well known socket address or port it may not be possible to bind a socket to the required
	 * {@code SocketAddress} if there is a connection in the timeout state involving the socket address or port.
	 * <p>
	 * Enabling {@link SocketOptions#SO_REUSEADDR SO_REUSEADDR} prior to binding the socket using
	 * {@link #bind(SocketAddress)} allows the socket to be bound even though a previous connection is in a timeout
	 * state.
	 * <p>
	 * When a {@code Socket} is created the initial setting of {@link SocketOptions#SO_REUSEADDR SO_REUSEADDR} is
	 * disabled.
	 * <p>
	 * The behaviour when {@link SocketOptions#SO_REUSEADDR SO_REUSEADDR} is enabled or disabled after a socket is bound
	 * (See {@link #isBound()}) is not defined.
	 *
	 * @param on
	 *            whether to enable or disable the socket option
	 * @exception SocketException
	 *                if an error occurs enabling or disabling the {@link SocketOptions#SO_REUSEADDR SO_REUSEADDR}
	 *                socket option, or the socket is closed.
	 * @since 1.4
	 * @see #getReuseAddress()
	 * @see #bind(SocketAddress)
	 * @see #isClosed()
	 * @see #isBound()
	 */
	public void setReuseAddress(boolean on) throws SocketException {
		throw new RuntimeException();
	}

	/**
	 * Tests if {@link SocketOptions#SO_REUSEADDR SO_REUSEADDR} is enabled.
	 *
	 * @return a {@code boolean} indicating whether or not {@link SocketOptions#SO_REUSEADDR SO_REUSEADDR} is enabled.
	 * @exception SocketException
	 *                if there is an error in the underlying protocol, such as a TCP error.
	 * @since 1.4
	 * @see #setReuseAddress(boolean)
	 */
	public boolean getReuseAddress() throws SocketException {
		throw new RuntimeException();
	}

	/**
	 * Closes this socket.
	 * <p>
	 * Any thread currently blocked in an I/O operation upon this socket will throw a {@link SocketException}.
	 * <p>
	 * Once a socket has been closed, it is not available for further networking use (i.e. can't be reconnected or
	 * rebound). A new socket needs to be created.
	 *
	 * <p>
	 * Closing this socket will also close the socket's {@link java.io.InputStream InputStream} and
	 * {@link java.io.OutputStream OutputStream}.
	 *
	 * <p>
	 * If this socket has an associated channel then the channel is closed as well.
	 *
	 * @exception IOException
	 *                if an I/O error occurs when closing this socket.
	 * @revised 1.4
	 * @spec JSR-51
	 * @see #isClosed
	 */
	@Override
	public synchronized void close() throws IOException {
		throw new RuntimeException();
	}

	/**
	 * Converts this socket to a {@code String}.
	 *
	 * @return a string representation of this socket.
	 */
	@Override
	public String toString() {
		throw new RuntimeException();
	}

	/**
	 * Returns the connection state of the socket.
	 * <p>
	 * Note: Closing a socket doesn't clear its connection state, which means this method will return {@code true} for a
	 * closed socket (see {@link #isClosed()}) if it was successfuly connected prior to being closed.
	 *
	 * @return true if the socket was successfuly connected to a server
	 * @since 1.4
	 */
	public boolean isConnected() {
		throw new RuntimeException();
	}

	/**
	 * Returns the binding state of the socket.
	 * <p>
	 * Note: Closing a socket doesn't clear its binding state, which means this method will return {@code true} for a
	 * closed socket (see {@link #isClosed()}) if it was successfuly bound prior to being closed.
	 *
	 * @return true if the socket was successfuly bound to an address
	 * @since 1.4
	 * @see #bind
	 */
	public boolean isBound() {
		throw new RuntimeException();
	}

	/**
	 * Returns the closed state of the socket.
	 *
	 * @return true if the socket has been closed
	 * @since 1.4
	 * @see #close
	 */
	public boolean isClosed() {
		throw new RuntimeException();
	}
}
