/*
 * Copyright 2013-2019 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break IS2T warranties on the whole library.
 */
package java.util.logging;

/**
 * Level defines a set of standard logging levels. The Level objects are ordered thanks to integer values. Levels are
 * used to control logging output. Enabling logging at a given level also enables logging at all higher levels.
 * <p>
 * Clients should normally used the predefined Level constants provided by this class. These levels are, from highest to
 * lowest:
 * <ul>
 * <li>SEVERE
 * <li>WARNING
 * <li>INFO
 * <li>CONFIG
 * <li>FINE
 * <li>FINER
 * <li>FINEST
 * </ul>
 * There are also two special levels:
 * <ul>
 * <li>ALL to enable logging all messages.
 * <li>OFF to turn off logging.
 * </ul>
 */
public class Level {

	/*
	 * Integer value of the standard Levels, from highest to lowest.
	 */
	private static final int LEVEL_OFF = Integer.MAX_VALUE;
	private static final int LEVEL_SEVERE = 1000;
	private static final int LEVEL_WARNING = 900;
	private static final int LEVEL_INFO = 800;
	private static final int LEVEL_CONFIG = 700;
	private static final int LEVEL_FINE = 500;
	private static final int LEVEL_FINER = 400;
	private static final int LEVEL_FINEST = 300;
	private static final int LEVEL_ALL = Integer.MIN_VALUE;

	/*
	 * The standard Levels, from highest to lowest.
	 */
	/**
	 * OFF is a special level that can be used to turn off logging. This level is initialized to
	 * {@link Integer#MAX_VALUE}.
	 */
	public static final Level OFF = new Level("OFF", LEVEL_OFF); //$NON-NLS-1$

	/**
	 * SEVERE is a message level indicating a serious failure.
	 * <p>
	 * In general SEVERE messages should describe events that are of considerable importance and which will prevent
	 * normal program execution. They should be reasonably intelligible to end users and to system administrators. This
	 * level is initialized to {@value #LEVEL_SEVERE}.
	 */
	public static final Level SEVERE = new Level("SEVERE", LEVEL_SEVERE); //$NON-NLS-1$
	/**
	 * WARNING is a message level indicating a potential problem.
	 * <p>
	 * In general WARNING messages should describe events that will be of interest to end users or system managers, or
	 * which indicate potential problems. This level is initialized to {@value #LEVEL_WARNING}.
	 */
	public static final Level WARNING = new Level("WARNING", LEVEL_WARNING); //$NON-NLS-1$
	/**
	 * INFO is a message level for informational messages.
	 * <p>
	 * Typically INFO messages will be written to the console or its equivalent. So the INFO level should only be used
	 * for reasonably significant messages that will make sense to end users and system admins. This level is
	 * initialized to {@value #LEVEL_INFO}.
	 */
	public static final Level INFO = new Level("INFO", LEVEL_INFO); //$NON-NLS-1$
	/**
	 * CONFIG is a message level for static configuration messages.
	 * <p>
	 * CONFIG messages are intended to provide a variety of static configuration information, to assist in debugging
	 * problems that may be associated with particular configurations. For example, CONFIG message might include the CPU
	 * type, the graphics depth, the GUI look-and-feel, etc. This level is initialized to {@value #LEVEL_CONFIG}.
	 */
	public static final Level CONFIG = new Level("CONFIG", LEVEL_CONFIG); //$NON-NLS-1$
	/**
	 * FINE is a message level providing tracing information.
	 * <p>
	 * All of FINE, FINER, and FINEST are intended for relatively detailed tracing. The exact meaning of the three
	 * levels will vary between subsystems, but in general, FINEST should be used for the most voluminous detailed
	 * output, FINER for somewhat less detailed output, and FINE for the lowest volume (and most important) messages.
	 * <p>
	 * In general the FINE level should be used for information that will be broadly interesting to developers who do
	 * not have a specialized interest in the specific subsystem.
	 * <p>
	 * FINE messages might include things like minor (recoverable) failures. Issues indicating potential performance
	 * problems are also worth logging as FINE. This level is initialized to {@value #LEVEL_FINE}.
	 */
	public static final Level FINE = new Level("FINE", LEVEL_FINE); //$NON-NLS-1$
	/**
	 * FINER indicates a fairly detailed tracing message. By default logging calls for entering, returning, or throwing
	 * an exception are traced at this level. This level is initialized to {@value #LEVEL_FINER}.
	 */
	public static final Level FINER = new Level("FINER", LEVEL_FINER); //$NON-NLS-1$
	/**
	 * FINEST indicates a highly detailed tracing message. This level is initialized to {@value #LEVEL_FINEST}.
	 */
	public static final Level FINEST = new Level("FINEST", LEVEL_FINEST); //$NON-NLS-1$
	/**
	 * ALL indicates that all messages should be logged. This level is initialized to {@link Integer#MIN_VALUE}.
	 */
	public static final Level ALL = new Level("ALL", LEVEL_ALL); //$NON-NLS-1$

	// Name of the level object
	private final String name;

	// Integer value of the Level object
	private final int value;

	/**
	 * Creates a named Level with the given integer value.
	 * 
	 * @param name
	 *            the name of the level. For example "SEVERE".
	 * @param value
	 *            must be one the value above.
	 */
	protected Level(String name, int value) {
		this.name = name;
		this.value = value;
	}

	/**
	 * Return the name of the level.
	 * 
	 * @return the name.
	 */
	public String getName() {
		return name;
	}

	/**
	 * Return the integer value of the level. This value is useful to compare to level objects. The greater this number
	 * is, the highest the level is.
	 * 
	 * @return the integer value for this level.
	 */
	public int intValue() {
		return value;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		return name;
	}
}
